/*
* linterval.c
* interval arithmetic library for Lua 5.1 based on fi_lib
* Luiz Henrique de Figueiredo <lhf@tecgraf.puc-rio.br>
* 17 Jul 2018 09:59:57
* This code is hereby placed in the public domain.
*/

#include "fi_lib.h"

#include "lua.h"
#include "lauxlib.h"

#define MYNAME		"interval"
#define MYVERSION	MYNAME " library for " LUA_VERSION " / Jul 2018 / "\
			"using fi_lib"
#define MYTYPE		MYNAME

typedef interval *Interval;
static lua_State *LL=NULL;

#include "q_errm.c"

static double q_ipow(double x, int n)
{
 double r;
 for (r=1.0; n>0; n>>=1)
 {
  if (n&1) r*=x;
  x*=x;
 }
 return r;
}

static void Pset(Interval x, double a, double b)
{
 x->INF=a;
 x->SUP=b;
}

static void Pout(Interval x, double a, double b)
{
 x->INF=q_pred(a);
 x->SUP=q_succ(b);
}

static Interval Pnew(lua_State *L)
{
 Interval x=lua_newuserdata(L,sizeof(interval));
 luaL_getmetatable(L,MYTYPE);
 lua_setmetatable(L,-2);
 return x;
}

static Interval Pget(lua_State *L, int i, Interval x)
{
 LL=L;
 switch (lua_type(L,i))
 {
  case LUA_TNUMBER:
  {
   lua_Number a=lua_tonumber(L,i);
   Pset(x,a,a);
   return x;
  }
  case LUA_TSTRING:
  {
   lua_Number a=lua_tonumber(L,i);
   Pout(x,a,a);
   return x;
  }
  default:
   return luaL_checkudata(L,i,MYTYPE);
 }
 return NULL;
}

static int Ltostring(lua_State *L)		/** tostring(x) */
{
 interval A;
 Interval a=Pget(L,1,&A);
 lua_pushfstring(L,"[%f,%f]",a->INF,a->SUP);
 return 1;
}

static int Lnew(lua_State *L)			/** new(a,b,[outward]) */
{
 lua_Number a=luaL_checknumber(L,1);
 lua_Number b=luaL_optnumber(L,2,a);
 int outward=lua_toboolean(L,3);
 Interval x=Pnew(L);
 if (outward) Pout(x,a,b); else Pset(x,a,b);
 return 1;
}

static int Lextremes(lua_State *L)		/** extremes(x) */
{
 interval A;
 Interval a=Pget(L,1,&A);
 lua_pushnumber(L,a->INF);
 lua_pushnumber(L,a->SUP);
 return 2;
}

static int Ldiam(lua_State *L)			/** diam(x) */
{
 interval A;
 Interval a=Pget(L,1,&A);
 lua_pushnumber(L,q_diam(*a));
 return 1;
}

static int Linf(lua_State *L)			/** inf(x) */
{
 interval A;
 Interval a=Pget(L,1,&A);
 lua_pushnumber(L,a->INF);
 return 1;
}

static int Lsup(lua_State *L)			/** sup(x) */
{
 interval A;
 Interval a=Pget(L,1,&A);
 lua_pushnumber(L,a->SUP);
 return 1;
}

static int Lmid(lua_State *L)			/** mid(x) */
{
 interval A;
 Interval a=Pget(L,1,&A);
 lua_pushnumber(L,q_mid(*a));
 return 1;
}

static int Lneg(lua_State *L)			/** neg(x) */
{
 interval A;
 Interval a=Pget(L,1,&A);
 double p=a->INF;
 double q=a->SUP;
 Interval r=Pnew(L);
 Pset(r,-q,-p);
 return 1;
}

static int Lpow(lua_State *L)			/** pow(x,n) */
{
 interval A;
 Interval a=Pget(L,1,&A);
 int n=luaL_checkinteger(L,2);
 double p=a->INF;
 double q=a->SUP;
 double P,Q;
 Interval r=Pnew(L);
 if (n<0)
 {
  fi_abort(INV_ARG,-1);
  return 0;
 }
 else if (n==0)
  Pset(r,1.0,1.0);
 else if (n==1)
 {
  lua_settop(L,1);
  return 1;
 }
 else if (n==2)
 {
  *r=j_sqr(*a);
  return 1;
 }
 P=q_ipow(p,n);
 Q=q_ipow(q,n);
 if (n%2==1 || p>=0.0)
  Pout(r,P,Q);
 else if (q<0.0)
  Pout(r,Q,P);
 else
 {
  P=q_succ(P);
  Q=q_succ(Q);
  Pset(r,0.0,q_max(P,Q));
 }
 return 1;
}

static int Lcontains(lua_State *L)		/** contains(x,t) */
{
 interval A;
 Interval a=Pget(L,1,&A);
 lua_Number t=luaL_checknumber(L,2);
 lua_pushboolean(L,in_di(t,*a));
 return 1;
}

static int Lcontained(lua_State *L)		/** contained(x,y) */
{
 interval A,B;
 Interval a=Pget(L,1,&A);
 Interval b=Pget(L,2,&B);
 lua_pushboolean(L,in_ii(*a,*b));
 return 1;
}

static int Ldisjoint(lua_State *L)		/** disjoint(x,y) */
{
 interval A,B;
 Interval a=Pget(L,1,&A);
 Interval b=Pget(L,2,&B);
 lua_pushboolean(L,dis_ii(*a,*b));
 return 1;
}

static int Leq(lua_State *L)
{
 interval A,B;
 Interval a=Pget(L,1,&A);
 Interval b=Pget(L,2,&B);
 lua_pushboolean(L,ieq_ii(*a,*b));
 return 1;
}

static int Llt(lua_State *L)
{
 interval A,B;
 Interval a=Pget(L,1,&A);
 Interval b=Pget(L,2,&B);
 lua_pushboolean(L,a->SUP < b->INF);
 return 1;
}

static int Lle(lua_State *L)
{
 interval A,B;
 Interval a=Pget(L,1,&A);
 Interval b=Pget(L,2,&B);
 lua_pushboolean(L,a->SUP <= b->INF);
 return 1;
}

static int Pdo1(lua_State *L, interval (*f)(interval a))
{
 interval A;
 Interval a=Pget(L,1,&A);
 Interval r=Pnew(L);
 *r=f(*a);
 return 1;
}

static int Pdo2(lua_State *L, interval (*f)(interval a, interval b))
{
 interval A,B;
 Interval a=Pget(L,1,&A);
 Interval b=Pget(L,2,&B);
 Interval r=Pnew(L);
 *r=f(*a,*b);
 return 1;
}

#define DO(d,f,F)	static int L##f(lua_State *L) { return d(L,F); }
#define DO1(f,F)	DO(Pdo1,f,F)
#define DO2(f,F)	DO(Pdo2,f,F)

DO1(abs,j_abs)					/** abs(x) */
DO1(acos,j_acos)				/** acos(x) */
DO1(acosh,j_acsh)				/** acosh(x) */
DO1(acot,j_acot)				/** acot(x) */
DO1(acoth,j_acth)				/** acoth(x) */
DO1(asin,j_asin)				/** asin(x) */
DO1(asinh,j_asnh)				/** asinh(x) */
DO1(atan,j_atan)				/** atan(x) */
DO1(atanh,j_atnh)				/** atanh(x) */
DO1(cos,j_cos)					/** cos(x) */
DO1(cosh,j_cosh)				/** cosh(x) */
DO1(cot,j_cot)					/** cot(x) */
DO1(coth,j_coth)				/** coth(x) */
DO1(erf,j_erf)					/** erf(x) */
DO1(erfc,j_erfc)				/** erfc(x) */
DO1(exp,j_exp)					/** exp(x) */
DO1(exp10,j_ex10)				/** exp10(x) */
DO1(exp2,j_exp2)				/** exp2(x) */
DO1(expm1,j_expm)				/** expm1(x) */
DO1(log,j_log)					/** log(x) */
DO1(log10,j_lg10)				/** log10(x) */
DO1(log1p,j_lg1p)				/** log1p(x) */
DO1(log2,j_log2)				/** log2(x) */
DO1(sin,j_sin)					/** sin(x) */
DO1(sinh,j_sinh)				/** sinh(x) */
DO1(sqr,j_sqr)					/** sqr(x) */
DO1(sqrt,j_sqrt)				/** sqrt(x) */
DO1(tan,j_tan)					/** tan(x) */
DO1(tanh,j_tanh)				/** tanh(x) */
DO2(add,add_ii)					/** add(x,y) */
DO2(div,div_ii)					/** div(x,y) */
DO2(join,hull)					/** join(x,y) */
DO2(meet,intsec)				/** meet(x,y) */
DO2(mul,mul_ii)					/** mul(x,y) */
DO2(sub,sub_ii)					/** sub(x,y) */

static const luaL_Reg R[] =
{
	{ "__add",	Ladd	},		/** __add(x,y) */
	{ "__div",	Ldiv	},		/** __div(x,y) */
	{ "__eq",	Leq	},		/** __eq(x,y) */
	{ "__le",	Lle	},		/** __le(x,y) */
	{ "__lt",	Llt	},		/** __lt(x,y) */
	{ "__mul",	Lmul	},		/** __mul(x,y) */
	{ "__pow",	Lpow	},		/** __pow(x,n) */
	{ "__sub",	Lsub	},		/** __sub(x,y) */
	{ "__tostring",	Ltostring},		/** __tostring(x) */
	{ "__unm",	Lneg	},		/** __unm(x) */
#define DECLARE(f)	{ #f, L##f },
DECLARE(abs)
DECLARE(acos)
DECLARE(acosh)
DECLARE(acot)
DECLARE(acoth)
DECLARE(add)
DECLARE(asin)
DECLARE(asinh)
DECLARE(atan)
DECLARE(atanh)
DECLARE(contained)
DECLARE(disjoint)
DECLARE(contains)
DECLARE(cos)
DECLARE(cosh)
DECLARE(cot)
DECLARE(coth)
DECLARE(diam)
DECLARE(div)
DECLARE(erf)
DECLARE(erfc)
DECLARE(exp)
DECLARE(exp10)
DECLARE(exp2)
DECLARE(expm1)
DECLARE(extremes)
DECLARE(inf)
DECLARE(join)
DECLARE(log)
DECLARE(log10)
DECLARE(log1p)
DECLARE(log2)
DECLARE(meet)
DECLARE(mid)
DECLARE(mul)
DECLARE(neg)
DECLARE(new)
DECLARE(pow)
DECLARE(sin)
DECLARE(sinh)
DECLARE(sqr)
DECLARE(sqrt)
DECLARE(sub)
DECLARE(sup)
DECLARE(tan)
DECLARE(tanh)
DECLARE(tostring)
	{ NULL,		NULL	}
};

LUALIB_API int luaopen_interval(lua_State *L)
{
 luaL_newmetatable(L,MYTYPE);
 lua_setglobal(L,MYNAME);
 luaL_register(L,MYNAME,R);
 lua_pushliteral(L,"version");			/** version */
 lua_pushliteral(L,MYVERSION);
 lua_settable(L,-3);
 lua_pushliteral(L,"__index");
 lua_pushvalue(L,-2);
 lua_settable(L,-3);
 return 1;
}
