/*
 * LUACOM.cpp
 *
 *   Bind de Lua com COM/OLE
 *
 *  Renato Cerqueira
 *  Vinicius Almendra
 *
 */

// RCS Info
static char *rcsid = "$Id: luacom.cpp,v 1.29 2003/05/26 17:57:37 almendra Exp $";
static char *rcsname = "$Name:  $";
static char *g_version = "1.0 (BETA)";

#include <string.h>
#include <iostream.h>
#include <stdlib.h>

#include <ole2.h>
#include <ocidl.h>

#include <assert.h>
#include <stdio.h>
#include <math.h>

extern "C"
{
#include "lua.h"
#include "lauxlib.h"
#include "luadebug.h"
}

#include "luabeans.h"
#include "luacom.h"
#include "tLuaDispatch.h"
#include "tLuaCOM.h"
#include "tLuaCOMException.h"
#include "tCOMUtil.h"
#include "tLuaCOMClassFactory.h"
#include "tLuaCOMTypeHandler.h"
#include "tLuaCOMConnPoints.h"

#include "LuaAux.h"
#include "tUtil.h"

// macro for simplifying exception handling

#define CHK_COM_ERR2(hr) CHK_COM_ERR(hr, tUtil::GetErrorMessage(hr))


// some string constants

#define GET_PREFIX "get"
#define PUT_PREFIX "set"


//////////////////////
//                    //
//  FUNCOES INTERNAS  //
//                    //
////////////////////////

/*
 * getLuaBeans
 *
 *  Retorna o objeto LuaBeans associado ao estado Lua fornecido 
 *
 */

LuaBeans *getLuaBeans(lua_State *L)
{
  return LuaBeans::getBeans(L, tLuaCOM::TAG);
}

static tLuaCOM* ImplInterface(ITypeLib* typelib,
                              ITypeInfo* coclasstypeinfo,
                              const char* interface_name,
                              int unlocked_ref,
                              LuaBeans* lbeans)
{
  CHECKPARAM(typelib && interface_name && lbeans);

  ITypeInfo* interface_typeinfo = 
    tCOMUtil::GetInterfaceTypeInfo(typelib, interface_name);

  if(interface_typeinfo == NULL)
    return NULL;

  tLuaDispatch* iluacom = 
    tLuaDispatch::CreateLuaDispatch(interface_typeinfo, unlocked_ref, lbeans);

  if(iluacom == NULL)
  {
    COM_RELEASE(interface_typeinfo);
    return NULL;
  }

  tLuaCOM *lcom = NULL;

  try
  {
    lcom = tLuaCOM::CreateLuaCOM(iluacom, coclasstypeinfo, lbeans);
  }
  catch(class tLuaCOMException& e)
  {
    UNUSED(e);
    lcom = NULL;
  }

  COM_RELEASE(interface_typeinfo);
  COM_RELEASE(iluacom);

  return lcom;
}



///////////////////////////////////
//                               //
//  FUNCOES EXPORTADAS PARA LUA  //
//                               //
///////////////////////////////////

/*
 * luacom_ShowHelp
 *
 * Parametros: 
 *
 *  (1) objeto luacom
 *
 *   Mostra Help da type library associada ao objeto
 *   luacom
 */

static int luacom_ShowHelp(lua_State *L)
{
  char *pHelpFile = NULL; 
  unsigned long context = 0;

  tLuaCOM* luacom = (tLuaCOM *) getLuaBeans(L)->check_tag(1);

  luacom->getHelpInfo(&pHelpFile, &context);

  if(pHelpFile != NULL)
  {
    if(context != 0)
      WinHelp(NULL, pHelpFile, HELP_CONTEXT, context);
    else
      WinHelp(NULL, pHelpFile, HELP_FINDER, 0);
  }

  return 0;
}

/*
 * luacom_Connect
 *
 *   Parametros (Lua):
 *
 *    objeto luacom
 *    tabela que implementara a interface de conexao
 *
 *   Retorno
 *    objeto luacom que encapsula objeto luacom implementado pela
 *    tabela fornecida ou nil se nao for possvel estabelecer a
 *    conexao
 *
 *   Cria um connection point utilizando a tabela
 *   passada como parametro e o conecta com o objeto
 *   passado tambem como parametro
 */

static int luacom_Connect(lua_State *L)
{
  tLuaDispatch *server_disp = NULL;
  ITypeInfo *pTypeinfo = NULL;
  int ref = 0;
  LuaBeans *lbeans = getLuaBeans(L);

  // check parameters
  tLuaCOM* client = (tLuaCOM *) lbeans->check_tag(1);

  if(lua_type(L, 2) != LUA_TTABLE && lua_type(L, 2) != LUA_TUSERDATA)
  {
    luaL_argerror(L, 2, "Implementation must be a table or a userdata");
  }

  /* gets a unlocked reference to the implementation */
  lua_pushvalue(L, 2);
  int unlocked_ref = lua_ref(L, 0);

  pTypeinfo = client->GetDefaultEventsInterface();

  if(pTypeinfo == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  server_disp = 
    tLuaDispatch::CreateLuaDispatch(pTypeinfo, unlocked_ref, lbeans);

  if(server_disp == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  tLuaCOM* server = NULL;

  try
  {
    server = tLuaCOM::CreateLuaCOM(server_disp, client->GetCoClassTypeInfo(), lbeans);
  }
  catch(class tLuaCOMException& e)
  {
    UNUSED(e);
    server = NULL;
  }

  COM_RELEASE(server_disp);

  if(server == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  client->addConnection(server);

  lbeans->push(server); 

  return 1;
}

/*
 *  luacom_ImplInterfaceFromTypelib (Lua2C)
 *
 *  Parametros:
 *    1. Tabela de implementacao
 *    2. Nome da type library
 *    3. Nome da interface a ser implementada
 *    4. (Opcional) Nome da CoClass da qual a interface faz parte
 *       necessario no caso de se querer expor objetos implementados
 *       com essa interface ou no caso de se querer localizar uma
 *       interface source (eventos)
 *
 *  Retorno:
 *    1. Objeto LuaCOM que encapsula a implementacao via Lua
 *        ou nil em caso de erro
 *
 *  Implementa uma interface descrida em uma type library
 *  dada
 */

static int luacom_ImplInterfaceFromTypelib(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  if(lua_type(L, 1) != LUA_TTABLE && lua_type(L,1) != LUA_TUSERDATA)
  {
    luaL_argerror(L, 1, "Implementation must be a table or a userdata");
  }

  lua_pushvalue(L, 1);
  const int unlocked_ref = lua_ref(L, 0);

  const char* typelib_name = luaL_check_lstr(L, 2, NULL);
  const char* pcInterface = luaL_check_lstr(L, 3, NULL);
  const char* coclassname = luaL_opt_lstr(L, 4, NULL, NULL);

  // gets typelib
  ITypeLib* typelib = tCOMUtil::LoadTypeLibByName(typelib_name);

  if(!typelib)
  {
    lua_pushnil(L);
    return 1;
  }

  // gets coclass typeinfo
  ITypeInfo *coclassinfo = NULL;

  if(coclassname)
  {
    coclassinfo = tCOMUtil::GetCoClassTypeInfo(typelib, coclassname);

    if(!coclassinfo)
    {
      typelib->Release();
      lua_pushnil(L);
      return 1;
    }
  }

  tLuaCOM* lcom = 
    ImplInterface(typelib, coclassinfo, pcInterface, unlocked_ref, lbeans);

  if(coclassinfo != NULL)
    COM_RELEASE(coclassinfo);

  COM_RELEASE(typelib);

  if(!lcom)
  {
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom); 
  return 1;
}

/* 
 *  luacom_ImplInterface (Lua2C)
 *
 *  Cria uma implementacao IDispatch para um
 *  objeto Lua dado um ProgID e o nome da
 *  interface
 *
 *  In: Implementation: table,
 *      ProgID: string,
 *      Interface: string,
 *
 *  Out: LuaCOM_obj (table with tag LuaCOM)
*/

static int luacom_ImplInterface(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  // gets parameters
  if(lua_type(L, 1) != LUA_TTABLE && lua_type(L,1) != LUA_TUSERDATA)
  {
    luaL_argerror(L, 1, "Implementation must be a table or a userdata");
  }

  // pushes lua table on top of stack
  lua_pushvalue(L, 1);
  const int unlocked_ref = lua_ref(L, 0);

  const char* pcProgID = luaL_check_lstr(L, 2, NULL);
  const char* pcInterface = luaL_check_lstr(L, 3, NULL);

  // gets typelib
  CLSID clsid;

  if(tCOMUtil::ProgID2CLSID(&clsid, pcProgID) != S_OK)
  {
    lua_pushnil(L);
    return 1;
  }

  ITypeLib* typelib = tCOMUtil::LoadTypeLibFromCLSID(clsid);

  if(typelib == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  // gets coclass typeinfo (we don't check return value
  // because NULL is a valid value

  ITypeInfo* coclassinfo = tCOMUtil::GetCoClassTypeInfo(typelib, clsid);

  tLuaCOM* lcom =
    ImplInterface(typelib, coclassinfo, pcInterface, unlocked_ref, lbeans);

  COM_RELEASE(coclassinfo);
  COM_RELEASE(typelib);

  if(lcom == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom);

  return 1;
}

/*
 *  luacom_CLSIDfromProgID
 *  Retorna string contendo CLSID associado a um ProgID
 */

static int luacom_CLSIDfromProgID(lua_State *L)
{
   const char* str = luaL_check_lstr(L, 1, NULL);
   wchar_t* clsid_str;
   wchar_t* progId;
   CLSID clsid;
   HRESULT hr;

   progId = (wchar_t*) malloc( (strlen(str) + 1) * sizeof(wchar_t));
   mbstowcs(progId,str,strlen(str)+1);

   hr = CLSIDFromProgID(progId, &clsid);
   free(progId);
   if (FAILED(hr))
   {
      cout << "Error looking for class " << str << endl;
      lua_pushnil(L);
      return 1;
   }
   hr = StringFromCLSID(clsid, &clsid_str);
   if (FAILED(hr))
   {
      cout << "Error looking for prog_id" << endl;
      lua_pushnil(L);
      return 1;
   }

   char* id_str = (char*) malloc( (wcslen(clsid_str) + 1) * sizeof(char));
   wcstombs(id_str,clsid_str,wcslen(clsid_str)+1);
   lua_pushstring(L, id_str);
   free(id_str);

   return 1;
}

/*
 *  luacom_ProgIDfromCLSID
 *  Retorna string contendo o ProgID associado a um CLSID
 */

static int luacom_ProgIDfromCLSID(lua_State *L)
{
   const char* str = luaL_check_lstr(L, 1, NULL);
   wchar_t* clsid_str;
   LPOLESTR progId;
   CLSID clsid;
   HRESULT hr;

   clsid_str = (wchar_t*) malloc( (strlen(str) + 1) * sizeof(wchar_t));
   mbstowcs(clsid_str,str,strlen(str)+1);

   hr = CLSIDFromString(clsid_str, &clsid);
   free(clsid_str);
   if (FAILED(hr))
   {
      cout << "Error looking for class " << str << endl;
      lua_pushnil(L);
      return 1;
   }
   hr = ProgIDFromCLSID(clsid, &progId);
   if (FAILED(hr))
   {
      cout << "Error looking for prog_id"<< endl;
      lua_pushnil(L);
      return 1;
   }

   char* id_str = (char*) malloc( (wcslen(progId) + 1) * sizeof(char));
   wcstombs(id_str,progId,wcslen(progId)+1);
   lua_pushstring(L, id_str);
   free(id_str);

   return 1;
}

/*
 *  luacom_CreateObject
 *  Retorna um objeto LuaCOM que instancia o objeto
 *  COM identificado pelo ProgID dado
 */

static int luacom_CreateObject(lua_State *L)
{
  HRESULT hr       = S_OK;
  LPDISPATCH pdisp = NULL;
  LuaBeans *lbeans = getLuaBeans(L);

  const char *progId = luaL_check_lstr(L, 1, NULL);

  CLSID clsid;
  hr = tCOMUtil::ProgID2CLSID(&clsid, progId);

  if (FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  hr = CoCreateInstance(clsid,NULL,CLSCTX_ALL, IID_IDispatch,(void**)&pdisp);

  if (FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  // gets coclasstypeinfo
  ITypeInfo* coclassinfo = tCOMUtil::GetCoClassTypeInfo(pdisp, clsid);

  // we don't check coclassinfo, because NULL is a valid value

  tLuaCOM* lcom = NULL;
  
  try
  {
    lcom = tLuaCOM::CreateLuaCOM(pdisp, coclassinfo, lbeans);
  }
  catch(class tLuaCOMException& e)
  {
    UNUSED(e);
    lcom = NULL;
  }

  COM_RELEASE(pdisp);
  COM_RELEASE(coclassinfo);

  if(lcom == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom);

  return 1;
}


/*
 *  luacom_GetObject
 *  Cria um objeto LuaCOM associado a uma instancia
 *  ja' existente do objeto COM identificado pelo 
 *  ProgID dado
 */

static int luacom_GetObject(lua_State *L)
{
  HRESULT hr       = S_OK;
  IDispatch* pdisp = NULL;
  IUnknown* punk = NULL;
  LuaBeans *lbeans = getLuaBeans(L);

  const char *progId = luaL_check_lstr(L, 1, NULL);

  CLSID clsid;
  hr = tCOMUtil::ProgID2CLSID(&clsid, progId);

  if (FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  hr = GetActiveObject(clsid,NULL,&punk);

  if (FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  hr = punk->QueryInterface(IID_IDispatch, (void **) &pdisp);

  if(FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  punk->Release();
  punk = NULL;

  // gets coclasstypeinfo
  ITypeInfo* coclassinfo = tCOMUtil::GetCoClassTypeInfo(pdisp, clsid);

  tLuaCOM* lcom = NULL;

  try
  {
    lcom = tLuaCOM::CreateLuaCOM(pdisp, coclassinfo, lbeans);
  }
  catch(class tLuaCOMException& e)
  {
    UNUSED(e);
    lcom = NULL;
  }

  COM_RELEASE(pdisp);
  COM_RELEASE(coclassinfo);

  if(lcom == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom);

  return 1;
}

/*
 *  luacom_addConnection
 *  Associa um connection point de um objeto COM com
 *  outro objeto COM
 */

static int luacom_addConnection(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  tLuaCOM* client = (tLuaCOM*) lbeans->check_tag(1);
  tLuaCOM* server = (tLuaCOM*) lbeans->check_tag(2);

  bool result = client->addConnection(server);

  if(result == false)
    lua_pushnil(L);
  else
    lua_pushnumber(L, 1);

  return 1;
}

/*
 *  luacom_releaseConnection
 *  Desfaz connection point associado a um objeto LuaCOM
 */

static int luacom_releaseConnection(lua_State *L)
{
  tLuaCOM* lcom = (tLuaCOM*) getLuaBeans(L)->check_tag(1);

  lcom->releaseConnection();

  return 0;
}

//
// luacom_isMember
//
//  Informa se existe algum metodo ou propriedade com
//  o nome passado como parametro em lua
//

static int luacom_isMember(lua_State *L)
{
  // objeto luacom
  tLuaCOM* lcom = (tLuaCOM*) getLuaBeans(L)->check_tag(1);
  const char* member_name = luaL_check_lstr(L, 2, NULL);

  if(lcom->isMember(member_name))
    lua_pushnumber(L, 1);
  else
    lua_pushnil(L);

  return 1;
}

/*
 * luacom_NewObject
 *
 *  Creates a Component Object implemented in luacom
 *
 */

static int luacom_NewObject(lua_State *L)
{
  tLuaDispatch* iluacom           = NULL;
  ITypeLib* typelib               = NULL;
  ITypeInfo* interface_typeinfo   = NULL;
  ITypeInfo* coclassinfo          = NULL;
  tLuaCOM *lcom                   = NULL;
  HRESULT hr                      = S_OK;
  tLuaCOMConnPointContainer *cpc  = NULL;
  tLuaCOMConnPoint *cp            = NULL;

  LuaBeans *lbeans = getLuaBeans(L);

  // gets parameters

  if(lua_type(L, 1) != LUA_TTABLE && lua_type(L,1) != LUA_TUSERDATA)
  {
    luaL_argerror(L, 1, "Implementation must be a table or a userdata");
  }

  // pushes lua table on top of stack
  lua_pushvalue(L, 1);
  const int unlocked_ref = lua_ref(L, 0);

  const char* pcProgID = luaL_check_lstr(L, 2, NULL);

  try
  {
    // gets typelib
    CLSID clsid;

    hr = tCOMUtil::ProgID2CLSID(&clsid, pcProgID);
    CHK_COM_ERR2(hr);

    typelib = tCOMUtil::LoadTypeLibFromCLSID(clsid);
    CHK_LCOM_ERR((long)typelib, "Could not load type library.");
    

    // gets coclass typeinfo

    coclassinfo = tCOMUtil::GetCoClassTypeInfo(typelib, clsid);
    CHK_LCOM_ERR(coclassinfo, "CoClass not found in type library.");


    // gets the default interface typeinfo
    interface_typeinfo =
      tCOMUtil::GetDefaultInterfaceTypeInfo(coclassinfo, false);

    CHK_LCOM_ERR(interface_typeinfo, 
      "Could not find a suitable default interface.");


    // Creates IDispatch implementation
    iluacom = 
      tLuaDispatch::CreateLuaDispatch(interface_typeinfo, unlocked_ref, lbeans);


    // Creates associated luacom object
    lcom = tLuaCOM::CreateLuaCOM(iluacom, coclassinfo, lbeans);


    // Informs tLuaDispatch of coclassinfo (this allows implementation of
    // IProvideClassInfo[x]
    iluacom->SetCoClassinfo(coclassinfo);


    // Create connection points container and tells it to initialize
    // itself from coclassinfo and exports connection point for 
    // default source
    iluacom->BeConnectable(lbeans);

    if(iluacom->GetConnPointContainer())
    {
      cp = iluacom->GetConnPointContainer()->GetDefault();
    }
    else
      cp = NULL;
  }
  catch(class tLuaCOMException& e)
  {
    // releases pointers
    COM_RELEASE(iluacom);
    COM_RELEASE(interface_typeinfo);
    COM_RELEASE(coclassinfo);
    COM_RELEASE(typelib);

    // informs error with return value
    lua_pushnil(L);
    lua_pushnil(L);
    lua_pushstring(L, e.getMessage());
    return 3;
  }

  // returns LuaCOM object and connection point
  // for default source interface

  int retvals = 1;
  lbeans->push(lcom);

  if(cp)
  {
    cp->push();
    retvals++;
  }

  return retvals;
}


/*
 *  luacom_ExposeObject
 *
 *    Creates a class factory that exposes a
 *    COM object
 *
 *  Parameters:
 *
 *    1. LuaCOM object
 *
 *  Return values
 *
 *    1. Cookie to unexpose object
 *
 */

static int luacom_ExposeObject(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  // check parameters
  tLuaCOM* luacom = (tLuaCOM *) lbeans->check_tag(1);

  tLuaCOMClassFactory* luacom_cf = 
      new tLuaCOMClassFactory(luacom->GetIDispatch());

  ITypeInfo *coclassinfo = luacom->GetCoClassTypeInfo();
  
  CLSID clsid;
  HRESULT hr = tCOMUtil::GetCLSID(coclassinfo, &clsid);

  DWORD cookie = -1;

  hr = CoRegisterClassObject(
    clsid,
    luacom_cf,
    CLSCTX_LOCAL_SERVER,
    REGCLS_SINGLEUSE,
    &cookie);

  if(FAILED(hr))
  {
    delete luacom_cf;
    lua_pushnil(L);
    return 1;
  }

  lua_pushnumber(L, cookie);

  return 1;
}


/*
 *  luacom_RevokeObject
 *
 *    Revokes a previously registered class factory
 *
 *  Parameters:
 *
 *    1. Cookie
 *
 *  Return values
 *
 *    1. non-nil if succeeded
 *
 */

static int luacom_RevokeObject(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  // check parameters
  const int cookie = (int) luaL_check_number(L, 1);

  // revokes class object
  HRESULT hr = CoRevokeClassObject(cookie);

  if(FAILED(hr))
  {
    lua_pushnil(L);
  }
  else
  {
    lua_pushnumber(L, 1);
  }

  return 1;
}


/*
 *  luacom_RegisterObject
 *
 *    Registers a COM Object in the system registry
 *
 *  Parameters:
 *
 *    1. registration table or userdata
 *
 *  Return values
 *
 *    1. non-nil if successful
 *
 */


static int luacom_RegisterObject(lua_State *L)
{
  HRESULT hr = S_OK;
  const int bufsize = 1000;

  if(lua_type(L, 1) != LUA_TTABLE && lua_type(L,1) != LUA_TUSERDATA)
  {
    luaL_argerror(L, 1, "Registration info must be a table or userdata");
  }

  // gets the registration information from the registration table
  lua_pushstring(L, "VersionIndependentProgID");
  lua_gettable(L, 1);
  const char* VersionIndependentProgID = lua_tostring(L, -1);

  // gets the registration information from the registration table
  lua_pushstring(L, "ProgID");
  lua_gettable(L, 1);
  const char* ProgID = lua_tostring(L, -1);

  lua_pushstring(L, "TypeLib");
  lua_gettable(L, 1);
  const char* typelib_path = lua_tostring(L, -1);

  lua_pushstring(L, "CoClass");
  lua_gettable(L, 1);
  const char* CoClass = lua_tostring(L, -1);

  lua_pushstring(L, "ComponentName");
  lua_gettable(L, 1);
  const char* ComponentName= lua_tostring(L, -1);

  lua_pushstring(L, "Arguments");
  lua_gettable(L, 1);
  const char* arguments = lua_tostring(L, -1);

  if(!
    (ProgID &&
     typelib_path &&
     CoClass))
  {
    lua_pushnil(L);
    return 1;
  }

  // Loads and registers the typelib
  ITypeLib *typelib = NULL;

  {
    wchar_t wcTypelib_path[bufsize];
    
    mbstowcs(wcTypelib_path, typelib_path, strlen(typelib_path)+1);

    hr = LoadTypeLibEx(wcTypelib_path, REGKIND_REGISTER, &typelib);
  }

  if(FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  // Gets the type library version and LIBID

  char version[30];
  char libId[bufsize];

  {
    TLIBATTR *plibattr = NULL;

    typelib->GetLibAttr(&plibattr);

    // gets version
    sprintf(version, "%d.%d", plibattr->wMajorVerNum, plibattr->wMinorVerNum);

    // gets libid
    wchar_t *wcLibId = NULL;
    StringFromCLSID(plibattr->guid, &wcLibId);
    wcstombs(libId, wcLibId, wcslen(wcLibId)+1);
    CoTaskMemFree(wcLibId);

    typelib->ReleaseTLibAttr(plibattr);
  }

  // gets the CoClass TypeInfo to get the CLSID
  ITypeInfo *coclassinfo = tCOMUtil::GetCoClassTypeInfo(typelib, CoClass);

  if(coclassinfo == NULL)
  {
    typelib->Release();
    lua_pushnil(L);
    return 1;
  }

  // gets the CLSID
  
  char clsid[bufsize];
  
  {
    TYPEATTR* ptypeattr = NULL;
    wchar_t* wcClsid=  NULL;

    coclassinfo->GetTypeAttr(&ptypeattr);

    StringFromCLSID(ptypeattr->guid, &wcClsid);
    wcstombs(clsid, wcClsid,wcslen(wcClsid)+1);

    coclassinfo->ReleaseTypeAttr(ptypeattr);
    CoTaskMemFree(wcClsid);
  }

  COM_RELEASE(coclassinfo);
  COM_RELEASE(typelib);

  //// Now we have all the information needed to perform the registration

  // registers ProgID
  char ID[bufsize];
  char CLSID[bufsize];
  char ModulePath[bufsize];

  // Be safe with null strings in these stack-allocated strings.
  ID[0] = 0;
  CLSID[0] = 0;
  ModulePath[0] = 0;

  GetModuleFileName(
    NULL,
    ModulePath,
    sizeof(ModulePath));

  if(arguments)
  {
    strcat(ModulePath, " ");
    strcat(ModulePath, arguments);
  }

  // Create some base key strings.
  strcpy(CLSID, "CLSID\\");
  strcat(CLSID, clsid);

  // Create ProgID keys.
  tCOMUtil::SetRegKeyValue(
    ProgID,
    NULL,
    ComponentName);

  tCOMUtil::SetRegKeyValue(
    ProgID,
    "CLSID",
    clsid);

  // Create VersionIndependentProgID keys.
  tCOMUtil::SetRegKeyValue(
    VersionIndependentProgID,
    NULL,
    ComponentName);

  tCOMUtil::SetRegKeyValue(
    VersionIndependentProgID,
    "CurVer",
    ProgID);

  tCOMUtil::SetRegKeyValue(
    VersionIndependentProgID,
    "CLSID",
    clsid);

  // Create entries under CLSID.
  tCOMUtil::SetRegKeyValue(
    CLSID,
    NULL,
    ComponentName);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "ProgID",
    ProgID);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "VersionIndependentProgID",
    VersionIndependentProgID);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "LocalServer32",
    ModulePath);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "Version",
    version);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "TypeLib",
    libId);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "Programmable",
    NULL);



  // signals success
  lua_pushnumber(L, 1);
  return 1;
}



/*
 *  luacom_GetIUnknown
 *
 *    Returns the a IUnknown interface for a
 *    LuaCOM object
 *
 *  Parameters:
 *
 *    1. LuaCOM object
 *
 *  Return values
 *
 *    1. IUnknown pointer (a userdata)
 *
 */


static int luacom_GetIUnknown(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  // check parameters
  tLuaCOM* luacom = (tLuaCOM *) lbeans->check_tag(1);

  IDispatch* pdisp = luacom->GetIDispatch();
  IUnknown* punk = NULL;
  
  pdisp->QueryInterface(IID_IUnknown, (void **) &punk);

  // checks whether there is a usertag for this IUnknown
  // If exists, simply uses it

  // gets IUnknown tag
  lua_getglobal(L, LUACOM_IUNKNOWN_TAGNAME);
  const int IUnknown_tag = (int) lua_tonumber(L, -1);
  lua_remove(L, -1);

  // pushes userdata
  lua_pushusertag(L, punk, LUA_ANYTAG);

  if(lua_tag(L, -1) == IUnknown_tag)
  {
    // there was already a userdata for that pointer, 
    // so we have to decrement the reference count,
    // as the garbage collector will be called just
    // once for any userdata
    punk->Release();
  }
  else
  {
    // this is the first userdata with this value,
    // so corrects the tag
    lua_settag(L, IUnknown_tag);
  }

  return 1;
}






static int luacom_DumpTypeInfo(lua_State *L)
{
  tLuaDispatch *server_disp = NULL;
  ITypeInfo *pTypeinfo = NULL;
  LuaBeans *lbeans = getLuaBeans(L);

  // check parameters
  tLuaCOM* obj = (tLuaCOM *) lbeans->check_tag(1);

  if(obj->hasTypeInfo())
    tCOMUtil::DumpTypeInfo(obj->GetTypeInfo());

  return 0;
}



/*
 *  luacom_RegisterTypeConvFunction
 *
 *    Registers a type conversion callback for tags
 *    unknown to LuaCOM
 *
 *  Parameters:
 *
 *    1. function
 *
 *  Return values
 *
 *    None
 *
 */

#if 0
static int luacom_RegisterTypeConvFunction(lua_State *L)
{
  HRESULT hr = S_OK;

  luaL_checktype (L, -1, LUA_TFUNCTION);

  lua_setglobal(L, LUACOM_TYPECONV_FUNCTION);

  return 0;
}
#endif






//////////////////////////////////////
//                                  //
//  TAG METHODS DO USERTAG IUNKNOWN //
//                                  //
//////////////////////////////////////

/*
 * tag method que gerencia garbage collection
 */

static int IUnknown_tag_gc(lua_State *L)
{
   IUnknown* punk = (IUnknown*) lua_touserdata(L, 1);

   if(punk != NULL)
     punk->Release();

   return 0;
}


////////////////////////////////////
//                                //
//  TAG METHODS DO OBJETO LUACOM  //
//                                //
////////////////////////////////////

/*
 * tag method que gerencia garbage collection
 */

static int tagmeth_gc(lua_State *L)
{
   tLuaCOM* lcom = (tLuaCOM*) lua_touserdata(L, 1);

   assert(lcom);

   if(lcom != NULL && lcom->may_be_collected())
     delete lcom;

   return 0;
}

/*
 * tag method que gerencia atribuicoes a campos
 * do objeto luacom
 */

static int tagmeth_settable(lua_State *L)
{
  DISPID dispid;
  HRESULT hr              = S_OK;
  FUNCDESC* pfuncdesc     = NULL;
  const char* field_name  = NULL;

  /* indexes to the parameters coming from lua */
  const int table_param = 1; 
  const int index_param = 2;
  const int value_param = 3;

  LuaBeans *lbeans = getLuaBeans(L);

  try
  {
    tLuaCOM* lcom = (tLuaCOM*) lbeans->from_lua(table_param);
    CHECK(lcom, INTERNAL_ERROR);

    field_name = lua_tostring(L, index_param);

    if(!field_name)
      return 0;

    // Here we have two possible situations: the object
    // has type info or the object does not have

    if(lcom->hasTypeInfo())
    {
      FuncInfo funcinfo;
      bool found = lcom->getFUNCDESC(field_name, funcinfo);

      if(!found)
      {
        // this field does not belong to the COM object. Sets
        // the value in the lua table
        lua_rawset(L, table_param);

        return 0;
      }
      else if(lua_type(L, value_param) == LUA_TFUNCTION)
      {
        // Here we have a redefinition of the field to a lua
        // function
        lua_rawset(L, table_param);

        return 0;

      }
      else if(!funcinfo.propput)
      {
        // Redefinition of a read only field using a value (not a function)
        // is not allowed. We do this to avoid confusion, as the user might
        // inadvertently set the value of a read-only field and LuaCOM would
        // silently store it in the table, "fooling" the client about the
        // real value the field in the COM object. If the user really wishes
        // to redefine the field, he should use a lua function instead

        lua_error(L, "Trying to set a read-only field in a COM object.");

        return 0;
      }


      // does call
      return lcom->call(funcinfo.propput->memid,
                        INVOKE_PROPERTYPUT,
                        funcinfo.propput,
                        tLuaObjList(value_param, 1)
                        );

    }
    else
    {
      bool found = lcom->getDISPID(field_name, &dispid);

      // Checks whether COM object really have this field
      // and if the user is trying to redefine the field
      if(!found || lua_type(L, value_param) == LUA_TFUNCTION)
      {
        
        lua_rawset(L, table_param);

        return 0;
      }

      // does call
      return lcom->call(dispid,
                        INVOKE_PROPERTYPUT,
                        NULL,
                        tLuaObjList(value_param, 1)
                        );
    }
  }
  catch(class tLuaCOMException& e)
  {
    lua_error(L, e.getMessage());
  }

  return 0;
}


/*
 * Closure que gerencia chamadas de metodos
 *
 */
static int callhook(lua_State *L)
{
  tLuaCOM* lcom       = NULL;
  HRESULT hr          = S_OK;
  DISPID dispid       = -1;
  FUNCDESC *pfuncdesc = NULL;
  long invkind  = INVOKE_FUNC;


  LuaBeans *lbeans = getLuaBeans(L);
  int num_return_values = -1;

  // upvalues
  const int luacom_obj_param  = -4;
  const int dispid_param      = -3;
  const int invkind_param     = -2;
  const int funcdesc_param    = -1;

  // self param
  const int self_param  = 1;
  const int first_param = 2;

  // number of lua parameters, excluding upvalues and
  // the self param
  const int num_params = lua_gettop(L) - 5;

  // retrieves parameters from lua stack
  lcom      = (tLuaCOM *) lua_touserdata(L, luacom_obj_param);
  dispid    = (DISPID) lua_tonumber(L, dispid_param);
  invkind   = (long) lua_tonumber(L, invkind_param);
  pfuncdesc = (FUNCDESC*) lua_touserdata(L, funcdesc_param);

  try
  {
    CHECK(lcom, INTERNAL_ERROR);
    
    // sets the parameter list excluding the 'self' param
    tLuaObjList& params = tLuaObjList(first_param, num_params);

    num_return_values = lcom->call(dispid, invkind, pfuncdesc, params);

    return num_return_values;
  }
  catch(class tLuaCOMException& e)
  {
    lua_error(L, e.getMessage());
    return 0;
  }
}

static bool checkPrefix(const char* pName, const char** ppStripped_name, bool* propget)
{
  if(strncmp(pName, GET_PREFIX, strlen(GET_PREFIX)) == 0)
  {
    *ppStripped_name = pName+strlen(GET_PREFIX);
    *propget = true;
    return true;
  }
  else if(strncmp(pName, PUT_PREFIX, strlen(PUT_PREFIX)) == 0)
  {
    *ppStripped_name = pName+strlen(PUT_PREFIX);
    *propget = false;
    return true;
  }
  else
    return false;
}



static int untyped_tagmeth_index(lua_State *L,
                                 tLuaCOM* lcom,
                                 const char *field_name)
{
  DISPID dispid;
  HRESULT hr         = S_OK;
  INVOKEKIND invkind = INVOKE_FUNC;


  // tries to get the DISPID
  {
    // positions for parameters received from Lua
    const int table_param = lua_gettop(L)-1;
    const int index_param = lua_gettop(L);

    bool found = lcom->getDISPID(field_name, &dispid);

    const char *stripped_name = NULL;
    bool is_propget = false;

    if (!found && checkPrefix(field_name, &stripped_name, &is_propget)) 
    {

      // checks for field redefinition
      {
        lua_pushstring(L, stripped_name);
        lua_rawget(L, table_param);

        if(lua_type(L, -1) == LUA_TFUNCTION)
        {
          // field has been redefined. Leaves function on top
          // of stack and returns
          return 1;
        }
        else  // no redefinition. Removes the value and continues
          lua_remove(L, -1);
      }

      // the user specified a property get. Tries to 
      // get the DISPID again, skipping the prefix

      found = lcom->getDISPID(stripped_name, &dispid);

      if(found)
      {
        if(is_propget)
          invkind = INVOKE_PROPERTYGET;
        else
          invkind = INVOKE_PROPERTYPUT;
      }
    }

    if(!found)
      return 0;
  }

  // Decides which INVOKEKIND should be used
  {

    switch(invkind)
    {
    case INVOKE_PROPERTYGET:
      {
        // pushes closure
        lua_pushuserdata(L, (void *) lcom);
        lua_pushnumber(L, dispid);
        lua_pushnumber(L, INVOKE_PROPERTYGET);
        lua_pushnumber(L, 0); // no funcdesc

        lua_pushcclosure(L, callhook, 4);
        return 1;
      }
      
    case INVOKE_PROPERTYPUT:
      {
        // pushes closure
        lua_pushuserdata(L, (void *) lcom);
        lua_pushnumber(L, dispid);
        lua_pushnumber(L, INVOKE_PROPERTYPUT);
        lua_pushnumber(L, 0); // no funcdesc

        lua_pushcclosure(L, callhook, 4);
        return 1;
      }

    default:
      {
        // pushes closure
        lua_pushuserdata(L, (void *) lcom);
        lua_pushnumber(L, dispid);
        lua_pushnumber(L, INVOKE_PROPERTYGET | INVOKE_FUNC);
        lua_pushnumber(L, 0); // no funcdesc

        lua_pushcclosure(L, callhook, 4);
        return 1;
      }
    }
  }
}



static int typed_tagmeth_index(lua_State *L,
                               tLuaCOM* lcom,
                               const char *field_name)
{
  HRESULT hr          = S_OK;
  bool is_propget = false;
  bool is_propput = false;
  FuncInfo funcinfo;


  // Now tries to get the FUNCDESC
  {
    // positions for parameters received from Lua
    const int table_param = lua_gettop(L)-1;
    const int index_param = lua_gettop(L);

    // First, assumes the user supplied the right name
    // of the method

    bool found = lcom->getFUNCDESC(field_name, funcinfo);
    const char *stripped_name = NULL;

    // if name not found, check for prefixes and for field
    // redefinition

    if(!found && checkPrefix(field_name, &stripped_name, &is_propget))
    {
      // it seems the user supplied a prefix. Check that
      is_propput = !is_propget;

      // checks for field redefinition
      {
        lua_pushstring(L, stripped_name);
        lua_rawget(L, table_param);

        if(lua_type(L, -1) == LUA_TFUNCTION)
        {
          // field has been redefined. Leaves function on top
          // of stack and returns
          return 1;
        }
        else  // no redefinition. Removes the value and continues
          lua_remove(L, -1);
      }


      // Now tries to get the right FUNCDESC, using the name
      // without the prefix

      if(is_propget)
      {
        // the user specified a property get. Tries to 
        // get the FUNCDESC again, skipping the prefix
        found = lcom->getFUNCDESC(stripped_name, funcinfo);
      }
      else // there are only two prefixes...
      {
        found = lcom->getFUNCDESC(stripped_name, funcinfo);
      }
    }

    if(!found)
      return 0;
  }


  // Tries to decide which invokekind should be used
  {
    // First, tests whether should be a property get

    if(funcinfo.propget && !is_propput)
    {
      // if it has no params and the user did not specify
      // a propertyput, make the call. If it's not, tests
      // whether might be a propertyget with params

      if(funcinfo.propget->cParams == 0 && !is_propget)
      {
        return lcom->call(funcinfo.propget->memid,
                          INVOKE_PROPERTYGET,
                          funcinfo.propget,
                          tLuaObjList()
                          );
      }
      else if(funcinfo.propget->cParams > 0 || is_propget)
      {
        // pushes closure
        lua_pushuserdata(L, (void *) lcom);
        lua_pushnumber(L, funcinfo.propget->memid);
        lua_pushnumber(L, INVOKE_PROPERTYGET);
        lua_pushuserdata(L, (void *) funcinfo.propget);

        lua_pushcclosure(L, callhook, 4);
        return 1;
      }
    }

    if(funcinfo.propput && !is_propget)
    {
      if(funcinfo.propput->cParams > 0)
      {
        // property put with parameters
        // pushes closure
        lua_pushuserdata(L, (void *) lcom);
        lua_pushnumber(L, funcinfo.propput->memid);
        lua_pushnumber(L, funcinfo.propput->invkind);
        lua_pushuserdata(L, (void *) funcinfo.propput);

        lua_pushcclosure(L, callhook, 4);
        return 1;
      }
    }

    if(funcinfo.func && !is_propget && !is_propput)
    {
      // pushes closure
      lua_pushuserdata(L, (void *) lcom);
      lua_pushnumber(L, funcinfo.func->memid);
      lua_pushnumber(L, INVOKE_FUNC);
      lua_pushuserdata(L, (void *) funcinfo.func);

      lua_pushcclosure(L, callhook, 4);
      return 1;
    }

    // no match: nothing found
    return 0;
  }
}



/*
 * tag method que gerencia leituras de campos
 * do objeto luacom
 *
 * In: table, index
 */

static int tagmeth_index(lua_State *L)
{
  // used variables
  tLuaCOM* lcom    = NULL;
  LuaBeans *lbeans = NULL;
  int retval       = 1;
  const char *field_name = NULL;

  try
  {
    // get LuaBeans
    lbeans = getLuaBeans(L);

    // indexes for the parameters in the Lua stack
    const int table_param = 1;
    const int index_param = 2; 

    // retrieves LuaCOM object
    lcom = (tLuaCOM*) lbeans->from_lua(table_param);
    CHECK(lcom, INTERNAL_ERROR);

    // retrieves the field name
    field_name = lua_tostring(L, index_param);

    if(!field_name)
      return 0;


    // Further processing will be done by
    // different functions, depending on
    // the presence of type information

    switch(lcom->hasTypeInfo())
    {
    case true:
      retval = typed_tagmeth_index(L, lcom, field_name);
      break;

    case false:
      retval = untyped_tagmeth_index(L, lcom, field_name);
      break;
    }
  }
  catch(class tLuaCOMException& e)
  {
    switch(e.code)
    {
    case tLuaCOMException::INTERNAL_ERROR:
    case tLuaCOMException::PARAMETER_OUT_OF_RANGE:
    case tLuaCOMException::UNSUPPORTED_FEATURE:
    default:
      lua_error(L, e.getMessage());
      break;
    }

  }

  return retval;
}


/////////////////////////////////////////////
//                                         //
//  TABELA DAS FUNCOES EXPORTADAS PARA LUA //
//                                         //
/////////////////////////////////////////////

static struct luaL_reg functions_tb []= 
{
  {"luacom_CreateObject",luacom_CreateObject},
  {"luacom_GetObject",luacom_GetObject},
  {"luacom_CLSIDfromProgID",luacom_CLSIDfromProgID},
  {"luacom_ImplInterface",luacom_ImplInterface},
  {"luacom_ImplInterfaceFromTypelib",luacom_ImplInterfaceFromTypelib},
  {"luacom_addConnection", luacom_addConnection},
  {"luacom_releaseConnection", luacom_releaseConnection},
  {"luacom_ProgIDfromCLSID",luacom_ProgIDfromCLSID},
  {"luacom_isMember",luacom_isMember},
  {"luacom_Connect",luacom_Connect},
  {"luacom_ShowHelp",luacom_ShowHelp},
  {"luacom_NewObject", luacom_NewObject},
  {"luacom_ExposeObject", luacom_ExposeObject},
  {"luacom_RevokeObject", luacom_RevokeObject},
  {"luacom_RegisterObject", luacom_RegisterObject},
  {"luacom_GetIUnknown", luacom_GetIUnknown},
  {"luacom_DumpTypeInfo", luacom_DumpTypeInfo}
};
  
static struct luaL_reg tagmethods_tb[]= {{"index", tagmeth_index},
                                  {"settable",tagmeth_settable}};
static struct luaL_reg ud_tagmethods_tb[]= {{"gc", tagmeth_gc}};


/////////////////////////////////////
//                                 //
//  FUNCOES EXPORTADAS PARA C/C++  //
//                                 //
/////////////////////////////////////

/*
 *  luacom_IDispatch2LuaCOM
 *  Recebe um ponteiro void* que deve ser
 *  um ponteiro para uma implementacao de uma
 *  interface IDispatch. E' criado um objeto
 *  LuaCOM cuja implementacao e' dada por essa
 *  interface. O objeto criado e' colocado na
 *  pilha de Lua
 */

LUACOM_API int luacom_IDispatch2LuaCOM(lua_State *L, void *pdisp_arg)
{
  LuaBeans *lbeans = getLuaBeans(L);

  if(pdisp_arg == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  tLuaCOM* lcom = NULL;

  try
  {
    lcom = tLuaCOM::CreateLuaCOM((IDispatch*)pdisp_arg, NULL, lbeans);
  }
  catch(class tLuaCOMException& e)
  {
    UNUSED(e);
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom);

  return 1;
}

/*
 *  Inicializacao da biblioteca
 */

LUACOM_API void luacom_open(lua_State *L)
{
  if(L == NULL)
    return;

  luaL_openlib(L, functions_tb,sizeof(functions_tb)/sizeof(struct luaL_reg));

  LuaBeans *lbeans = LuaBeans::createBeans(L, tLuaCOM::TAG);
  lbeans->reg_tagmethods(tagmethods_tb,
                        sizeof(tagmethods_tb)/sizeof(struct luaL_reg));
  lbeans->reg_ud_tagmethods(ud_tagmethods_tb,
                          sizeof(ud_tagmethods_tb)/sizeof(struct luaL_reg));


  // Creates a lua tag for IUnknown pointes
  int IUnknown_tag = lua_newtag(L);

  // Creates a lua tag for Connection Point tables
  // and sets tag method
  int IConnectionPoint_tag = lua_newtag(L);

  // Sets a tag method for garbage collection
  lua_pushcfunction(L, IUnknown_tag_gc);
  lua_settagmethod(L, IUnknown_tag, "gc");


  // stores in the table
  lua_pushnumber(L, IUnknown_tag);
  lua_setglobal(L, LUACOM_IUNKNOWN_TAGNAME);

#ifdef LUACOMLOG
  tUtil::OpenLogFile("luacom_log.txt");
#endif
}

/*
 *  Fechamento da biblioteca
 */

LUACOM_API void luacom_close(lua_State *L)
{
#ifdef LUACOMLOG
//  tUtil::CloseLogFile();
#endif

}


/*
 * Helper for implementing automation servers with LuaCOM
 */

LUACOM_API int luacom_detectAutomation(lua_State *L, int argc, char *argv[])
{
  int automation_result = LUACOM_NOAUTOMATION;
  int lua_result = 0;

  int top = lua_gettop(L);

  // expects a table with the callback functions

  luaL_checktype (L, -1, LUA_TTABLE);

  // processes command line, looking for automation switches and
  // calling the appropriate callback functions

  while(argc--)
  {
    if(!strcmp(argv[argc], "/Automation"))
    {
      lua_pushstring(L, "StartAutomation");
      lua_gettable(L, -2);
      lua_result = lua_call(L, 0, 0);

      if(!lua_result)
        automation_result = LUACOM_AUTOMATION;
      else
        automation_result = LUACOM_AUTOMATION_ERROR;

      break;
    }
    else  if(!strcmp(argv[argc], "/Register"))
    {
      lua_pushstring(L, "Register");
      lua_gettable(L, -2);
      lua_result = lua_call(L, 0, 0);

      if(!lua_result)
        automation_result = LUACOM_REGISTER;
      else
        automation_result = LUACOM_AUTOMATION_ERROR;

      break;
    }
  }

  // cleans stack
  lua_settop(L, top);

  return automation_result;
}
