/*
 * tLuaCOM.cpp
 *
 *  Implementacao da classe tLuaCOM
 *
 * Renato Cerqueira
 * Vinicius Almendra
 */

// RCS Info
static char *rcsid = "$Id: tLuaCOM.cpp,v 1.21 2003/05/26 17:49:48 almendra Exp $";
static char *rcsname = "$Name:  $";

#include <iostream.h>
#include <crtdbg.h>

#include "tLuaCOM.h"
#include "tLuaDispatch.h"
#include "LuaAux.h"
#include "tUtil.h"
#include "tLuaCOMException.h"
#include "tCOMUtil.h"

char* tLuaCOM::TAG = "LuaCOM";
long tLuaCOM::NEXT_ID = 0;

// custom exception check
#define CHK_COM_ERR2(hr) CHK_COM_ERR(hr, tUtil::GetErrorMessage(hr))

tLuaCOM::tLuaCOM(IDispatch *pdisp_arg,
                 ITypeInfo *ptinfo_arg,
                 ITypeInfo *coclass_arg,
                 LuaBeans *lbeans,
                 bool p_can_be_collected)
{
  connection_point = NULL;
  connection_point_cookie = NULL;

  pdisp = pdisp_arg;
  pdisp->AddRef();

  ptinfo = ptinfo_arg;

  if(ptinfo)
  {
    // gets ITypeComp interface. If it's not
    // available, frees typeinfo, as we only
    // use ITypeComp

    HRESULT hr = ptinfo->GetTypeComp(&ptcomp);

    if(SUCCEEDED(hr))
      ptinfo->AddRef();
    else
    {
      ptinfo->Release();
      ptinfo = NULL;
    }
  }
  else
    ptcomp = NULL;

  can_be_collected = p_can_be_collected;

  coclassinfo = coclass_arg;
  
  if(coclassinfo != NULL)
    coclassinfo->AddRef();

  typehandler = new tLuaCOMTypeHandler(ptinfo_arg, lbeans);

  // cleans FuncInfo array
  int i;
  for(i = 0; i < MAX_FUNCINFOS; i++)
    pFuncInfo[i].name = NULL;

  ID = tLuaCOM::NEXT_ID++;
#ifdef LUACOMLOG
  {
    char msg[100];

    sprintf(msg, "%d:created", ID);
    tUtil::log("tLuaCOM",msg);
  }
#endif
}

tLuaCOM::~tLuaCOM()
{
  if(connection_point != NULL)
    releaseConnection();

  // frees funcinfos
  {
    long counter = 0;

    while(pFuncInfo[counter].name != NULL &&
      counter < MAX_FUNCINFOS)
    {
      free(pFuncInfo[counter].name);
      pFuncInfo[counter].name = NULL;

      ReleaseFuncDesc(pFuncInfo[counter].propget);
      ReleaseFuncDesc(pFuncInfo[counter].propput);
      ReleaseFuncDesc(pFuncInfo[counter].func);

      counter++;
    }
  }


  COM_RELEASE(ptcomp);
  COM_RELEASE(ptinfo);
  COM_RELEASE(coclassinfo);
  COM_RELEASE(pdisp);


  delete typehandler;
  typehandler = NULL;

#ifdef LUACOMLOG
  {
    char msg[100];

    sprintf(msg, "%d:destroyed", ID);
    tUtil::log("tLuaCOM",msg);
  }
#endif
}

bool tLuaCOM::getFUNCDESC(const char *name, FuncInfo& funcinfo)
{
  // First, tries to see we have the FUNCDESC's cached

  long counter = 0;

  for(counter = 0; counter < MAX_FUNCINFOS; counter++)
  {
    // when .name is NULL, there is no further information
    if(pFuncInfo[counter].name == NULL)
      break;

    if(strcmp(name, pFuncInfo[counter].name) == 0)
      break;
  }

  // checks whether funcinfo was found
  if(pFuncInfo[counter].name != NULL)
  {
    funcinfo = pFuncInfo[counter];
    return true;
  }

  // did not find, so gets type information through
  // ITypeComp

  HRESULT hr = S_OK;
  BINDPTR bindptr;
  DESCKIND desckind;
  BSTR wName;
  ITypeInfo *info = NULL;

  unsigned int dumb = 0;

  wName = tUtil::string2bstr(name);

  unsigned long lhashval = LHashValOfName(LOCALE_SYSTEM_DEFAULT, wName);

  hr = ptcomp->Bind(wName, lhashval, INVOKE_PROPERTYGET,
    &info, &desckind, &bindptr);
  
  if(FAILED(hr) || desckind == DESCKIND_NONE)
    funcinfo.propget = NULL;
  else
  {
    funcinfo.propget = bindptr.lpfuncdesc;
    info->Release();
  }

  hr = ptcomp->Bind(wName, lhashval, INVOKE_FUNC,
    &info, &desckind, &bindptr);
  
  if(FAILED(hr) || desckind == DESCKIND_NONE)
    funcinfo.func = NULL;
  else
  {
    funcinfo.func = bindptr.lpfuncdesc;
    info->Release();
  }


  hr = ptcomp->Bind(wName, lhashval, INVOKE_PROPERTYPUT,
    &info, &desckind, &bindptr);
  
  if(FAILED(hr) || desckind == DESCKIND_NONE)
    funcinfo.propput = NULL;
  else
  {
    funcinfo.propput = bindptr.lpfuncdesc;
    info->Release();
  }

  // if there is not propertyput, then tries propputref

  if(funcinfo.propput == NULL)
  {
    hr = ptcomp->Bind(wName, lhashval, INVOKE_PROPERTYPUTREF,
      &info, &desckind, &bindptr);

    if(FAILED(hr) || desckind == DESCKIND_NONE)
      funcinfo.propput = NULL;
    else
    {
      funcinfo.propput = bindptr.lpfuncdesc;
      info->Release();
    }
  }

  SysFreeString(wName);

  // If no type information found, returns NULL
  if(!funcinfo.propget && !funcinfo.propput && !funcinfo.func)
    return false;
  else if(counter < MAX_FUNCINFOS)
  {
    CHECKPRECOND(pFuncInfo[counter].name == NULL);

    pFuncInfo[counter].name = strdup(name);

    pFuncInfo[counter].propget  = funcinfo.propget;
    pFuncInfo[counter].propput  = funcinfo.propput;
    pFuncInfo[counter].func     = funcinfo.func;

    return true;
  }
  else
    return true;
}


bool tLuaCOM::getDISPID(const char* name, DISPID* dispid)
{
   HRESULT hr;
   wchar_t* w_name = (wchar_t*) malloc( (strlen(name) + 1) * sizeof(wchar_t));
   mbstowcs(w_name,name,strlen(name)+1);

   hr = pdisp->GetIDsOfNames(IID_NULL, &w_name, 1,
                          LOCALE_SYSTEM_DEFAULT,dispid);
   free(w_name);
   
   return SUCCEEDED(hr);
}

int tLuaCOM::call(DISPID dispid,
                  int invkind,
                  FUNCDESC *pfuncdesc,
                  tLuaObjList& params)
{ 
   HRESULT hr             = 0;
   unsigned int i         = 0;
   UINT ArgErr            = 0;
   
   // number of return values (required to interface with Lua)
   int num_retvals        = 0;

   DISPPARAMS dispparams;
   VARIANTARG result;
   EXCEPINFO excepinfo;
      
   VariantInit(&result);

   // fills DISPPARAMS structure, converting lua arguments
   // to COM parameters
   typehandler->fillDispParams(dispparams, pfuncdesc, params, invkind);

   hr = pdisp->Invoke(
     dispid,
     IID_NULL, 
     LOCALE_SYSTEM_DEFAULT,
     invkind,
     &dispparams,
     &result,
     &excepinfo,
     &ArgErr);

   if(SUCCEEDED(hr))
   {
     try
     {
       // pushes first return value (if any)
       // if there is no type information, assumes that
       // all functions have return value
       if((pfuncdesc && pfuncdesc->elemdescFunc.tdesc.vt != VT_VOID) ||
           (!pfuncdesc && result.vt != VT_EMPTY))
       {
         typehandler->com2lua(result);
         num_retvals++;
       }

       // pushes out values
       if(invkind & INVOKE_FUNC)
         num_retvals += typehandler->pushOutValues(dispparams);
     }
     catch(class tLuaCOMException& e)
     {
       UNUSED(e);

       typehandler->releaseVariants(&dispparams);
       VariantClear(&result);

       throw;
     }

     typehandler->releaseVariants(&dispparams);
     VariantClear(&result);

     return num_retvals;
   }
   else // Houve erro
   {
     // Limpa parametros
     typehandler->releaseVariants(&dispparams);

     if(hr == DISP_E_EXCEPTION) // excecoes
     {
       if(excepinfo.bstrDescription != NULL)
         COM_EXCEPTION(tUtil::bstr2string(excepinfo.bstrDescription));
       else if(excepinfo.wCode != 0)
         COM_EXCEPTION(tUtil::GetErrorMessage(excepinfo.wCode));
       else if(excepinfo.scode != 0)
         COM_EXCEPTION(tUtil::GetErrorMessage(excepinfo.scode));
       else
         COM_EXCEPTION("Unknown exception");
     }
     else // Erros comuns
       COM_ERROR(tUtil::GetErrorMessage(hr));
   }
}

bool tLuaCOM::addConnection(tLuaCOM *server)
{
  if(!server->hasTypeInfo())
    return false;

  HRESULT hr;
  IDispatch *pdisp_server = server->GetIDispatch();

  IConnectionPointContainer *pcpc = NULL;

  assert(connection_point == NULL);
  if(connection_point != NULL)
    return false;

  hr = pdisp->QueryInterface
    (
      IID_IConnectionPointContainer, (void **) &pcpc
    );

  if(FAILED(hr))
  {
    cout << "ERROR: Doesn't implement IConnectionPointContainer" << endl;
    return false;
  }

  {
    IID guid;

    server->GetIID(&guid);

    hr = pcpc->FindConnectionPoint(guid, &connection_point);
  }

  pcpc->Release();
  pcpc = NULL;

  if(FAILED(hr))
  {
    cout << "ERROR: Requested connection point not supported" << endl;

    return false;
  }

  hr = connection_point->Advise
    (
      (IUnknown *) pdisp_server, &connection_point_cookie
    );

  if(FAILED(hr))
  {
    cout << "ERROR: Unable to advise connection point" << endl;

    connection_point->Release();
    connection_point = NULL;

    return false;
  }

  return true;
}

void tLuaCOM::releaseConnection(void)
{
  assert(connection_point);
  if(connection_point == NULL)
    return;

  connection_point->Unadvise(connection_point_cookie);
  connection_point->Release();
  connection_point = NULL;
}

//
//  isMember
//
//    Informa se existe algum metodo ou propriedade com
//    o nome passado como parametro
//

bool tLuaCOM::isMember(const char * name)
{
  HRESULT hr;
  DISPID dumb_dispid;
  
  wchar_t* w_name = (wchar_t*) malloc( (strlen(name) + 1) * sizeof(wchar_t));

  assert(w_name);
  if(!w_name)
    return false;

  mbstowcs(w_name, name, strlen(name)+1);

  hr = pdisp->GetIDsOfNames(IID_NULL, &w_name, 1,
                        LOCALE_SYSTEM_DEFAULT, &dumb_dispid);
  free(w_name);
  w_name = NULL;

  if(!FAILED(hr))
    return true;
  else
    return false;
}


void tLuaCOM::getHelpInfo(char **ppHelpFile, unsigned long *pHelpContext)
{
  if(!hasTypeInfo())
  {
    *ppHelpFile = NULL;
    return;
  }


  ITypeLib *typelib = NULL;
  BSTR helpfile;
  HRESULT hr = S_OK;
  
  hr = ptinfo->GetDocumentation(-1, NULL, NULL, pHelpContext, &helpfile);

  if(FAILED(hr) || helpfile == NULL)
  {
    *ppHelpFile = NULL;
    return;
  }

  // Se nao conseguiu help contextna propria interface, tenta obte-lo
  // na type library
  if(*pHelpContext == 0)
  {
    unsigned int dumb_index = 0;
    unsigned long typelib_help_context = 0;
    BSTR helpfile_typelib;

    hr = ptinfo->GetContainingTypeLib(&typelib, &dumb_index);

    if(!FAILED(hr))
    {
      hr = typelib->GetDocumentation(-1, NULL, NULL,
        &typelib_help_context, &helpfile_typelib);

      if(!FAILED(hr))
      {
        SysFreeString(helpfile);

        helpfile = helpfile_typelib;
        *pHelpContext = typelib_help_context;
      }
    }
  }

  int str_size = SysStringLen(helpfile);
  *ppHelpFile = (char *) malloc((str_size + 1)*sizeof(wchar_t));
  wcstombs(*ppHelpFile, helpfile, str_size+1);

  SysFreeString(helpfile);
}



//
// CreateLuaCOM
//


tLuaCOM * tLuaCOM::CreateLuaCOM(IDispatch * pdisp,
                                ITypeInfo* coclassinfo,
                                LuaBeans *lbeans,
                                ITypeInfo* typeinfo,
                                bool can_be_collected)
{
  HRESULT hr = S_OK;

  CHECKPARAM(pdisp && lbeans);

  // tries to get some CoClass typeinfo
  if(coclassinfo == NULL)
    coclassinfo = tCOMUtil::GetCoClassTypeInfo((IUnknown*)pdisp);

  if(!typeinfo)
    typeinfo = tCOMUtil::GetDispatchTypeInfo(pdisp);

  tLuaCOM *lcom = 
    new tLuaCOM(pdisp, typeinfo, coclassinfo, lbeans, can_be_collected);

  return lcom;
}

ITypeInfo * tLuaCOM::GetDefaultEventsInterface()
{
  if(coclassinfo == NULL)
    return NULL;
  
  return tCOMUtil::GetDefaultInterfaceTypeInfo(coclassinfo, true);
}

void tLuaCOM::ReleaseFuncDesc(FUNCDESC * pfuncdesc)
{
  CHECKPRECOND(ptinfo);

  if(pfuncdesc)
    ptinfo->ReleaseFuncDesc(pfuncdesc);
}

IDispatch * tLuaCOM::GetIDispatch()
{
  return pdisp;
}

void tLuaCOM::GetIID(IID * piid)
{
  CHECKPRECOND(ptinfo);

  TYPEATTR *ptypeattr;

  ptinfo->GetTypeAttr(&ptypeattr);
  *piid = ptypeattr->guid;
  ptinfo->ReleaseTypeAttr(ptypeattr);
}


ITypeInfo* tLuaCOM::GetCoClassTypeInfo()
{
  return coclassinfo;
}


ITypeInfo* tLuaCOM::GetTypeInfo()
{
  return ptinfo;
}

bool tLuaCOM::may_be_collected()
{
  return can_be_collected;
}

bool tLuaCOM::hasTypeInfo(void)
{
  if(ptinfo)
    return true;
  else
    return false;
}
