--
-- LuaCOM package
--
--   Enhanced functionality for the LuaCOM library
--


-- startup code: presumes LuaCOM has already been initialized
-- and lies in the luacom table

assert(luacom)



function luacom.DumpDispinterface(typeinfo)
  assert(typeinfo.type == "dispinterface")

  typeinfo:Load()

  local function DumpInterface(i,v)

    local params = ""

    table.foreachi(v,
      function(index, param)
        if params ~= "" then params = params..", " end

        params = params..param.name
      end
    )

    if params ~= "" or v.type == "method" then
      params = "("..params..")"
    else
      params = ""
    end

    print(i, v.name..params)
    print("","["..v.type.."]")
    print("",v.description)
    print()
  end

  table.foreachi(typeinfo, DumpInterface)
end


luacom.RawDumpTypeInfo = luacom.DumpTypeInfo

function luacom.DumpTypeInfo(obj)
  assert(type(obj) == "table")

  if obj.type == "dispinterface" then
    luacom.DumpDispinterface(obj)
  end

  if obj.typeinfo then
    luacom.DumpTypeInfo(obj.typeinfo)
  end
end



function luacom.DumpTypeLib(typelib)

  assert(typelib)

  print(typelib.name)
  print(typelib.description)

  table.foreachi(typelib, 
    function(i,v)
      descr = string.format("%d: %-25s\t%-15s\t%s", i, v.name, v.type, v.description)
      print(descr)
    end
  )
end



function luacom._storeTypeInfo(obj)

  obj.typeinfo = luacom.GetTypeInfo(obj)

  if obj.typeinfo then
    obj.typeinfo.Dump = luacom.DumpDispinterface
    obj.typelib = obj.typeinfo.typelib
  end

  if obj.typelib then
    obj.typelib.Dump = luacom.DumpTypeLib

    function obj.typelib:LoadConstants()

      -- export constants
      table.foreachi(self,
        function(i,v)
          if v.public and v.type == "enumeration" then
            table.foreach(v.values,
              function(enum_tag, enum_value)
                _G[enum_tag] = enum_value
              end
            )
          end
        end
      )

    end -- LoadConstants

  end -- typelib

end


luacom.RawCreateObject = luacom.CreateObject

function luacom.CreateObject(ID, mode)
  local obj = luacom.RawCreateObject(ID, mode)

  if obj == nil then
    return obj
  end

  luacom._storeTypeInfo(obj)

  return obj
end

luacom.RawGetObject = luacom.GetObject

function luacom.GetObject(string)
  local obj = luacom.RawGetObject(string)

  if obj == nil then
    return obj
  end

  luacom._storeTypeInfo(obj)

  return obj
end



function luacom.CreateLocalObject(ID)
  return luacom.CreateObject(ID, "local_server")
end



function luacom.CreateInprocObject(ID)
  return luacom.CreateObject(ID, "inproc_server")
end


function luacom.pairs(luacom_obj)

  assert(luacom_obj)

  local enumerator = luacom.GetEnumerator(luacom_obj)

  if enumerator == nil then
    error("Could not get an enumerator")
    return
  end

  local function iterator(state, index)
    local value = state:Next()
    
    if value == nil then
      return nil
    else
      return index+1, value
    end

  end

  return iterator, enumerator, 0
end
