// tLuaTLB.cpp: implementation of the tLuaTLB class.
//
//////////////////////////////////////////////////////////////////////

#include "tLuaTLB.h"

#include "tCOMUtil.h"
#include "tLuaCOMException.h"
#include "tUtil.h"
#include "luacom_internal.h"
#include "LuaAux.h"

#include "tLuaCOM.h"
#include "tLuaCOMTypeHandler.h"

extern "C"
{
#include "luacompat.h"
}

// macro for simplifying exception handling

#define CHK_COM_ERR2(hr) CHK_COM_ERR(hr, tUtil::GetErrorMessage(hr))


const char tLuaTLB::type_name[] = "ITypeLib";
const char tLuaTLB::pointer_type_name[] = "ITypeLib_pointer";

const char tLuaTypeInfo::type_name[] = "ITypeInfo";
const char tLuaTypeInfo::pointer_type_name[] = "ITypeInfo_pointer";

const char 
  tLuaTypeInfo::Dispinterface::tLuaFuncDesc::type_name[] = "FUNCDESC";
const char 
  tLuaTypeInfo::Dispinterface::tLuaFuncDesc::pointer_type_name[] = "FUNCDESC_pointer";



//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

tLuaTLB::tLuaTLB(lua_State* L, ITypeLib *p_typelib)
{
  method_table.Add("ShowHelp", ShowHelp, FUNC);

  typelib = p_typelib;
  typelib->AddRef();
}

tLuaTLB::~tLuaTLB()
{
  COM_RELEASE(typelib);
}

int tLuaTLB::index(lua_State* L)
{
  return 0;
}

int tLuaTLB::pushNew(lua_State *L, ITypeLib *p_typelib)
{
  CHECKPRECOND(p_typelib);

  // tests whether we have a cached instance
  if(pushCachedObject(L, p_typelib))
    return 1;

  LUASTACK_SET(L);

  tLuaTLB* lua_tlb = new tLuaTLB(L, p_typelib);

  int retval = generic_PushNew(L, lua_tlb, type_name, pointer_type_name);
  cacheObject(L, (void*) p_typelib);

  // Fills table with static values
  BSTR name, description;

  p_typelib->GetDocumentation(-1, &name, &description, NULL, NULL);

  lua_pushstring(L, "name");
  lua_pushstring(L, tUtil::bstr2string(name));
  lua_settable(L, -3);

  lua_pushstring(L, "description");
  lua_pushstring(L, tUtil::bstr2string(description));
  lua_settable(L, -3);

  SysFreeString(name);
  SysFreeString(description);

  int count = p_typelib->GetTypeInfoCount();

  ITypeInfo* typeinfo = NULL;

  while(count)
  {
    p_typelib->GetTypeInfo(count-1, &typeinfo);

    // Creates typeinfo object, makes a copy
    // and stores it in the table
    tLuaTypeInfo* lua_typeinfo =
      dynamic_cast<tLuaTypeInfo*>(tLuaTypeInfo::pushNew(L, typeinfo));
    COM_RELEASE(typeinfo);

    lua_rawseti(L, -2, count);

    count--;
  }

  LUASTACK_CLEAN(L, 1);

  return retval;
}

int tLuaTLB::ShowHelp(tLuaObject* lua_obj, lua_State* L)
{
	tLuaTLB* lua_tlb =
    dynamic_cast<tLuaTLB*>(lua_obj);
  
  CHECKPRECOND(lua_tlb);

  HRESULT hr = S_OK;
  BSTR helpfile;
  ULONG helpcontext;

  hr = 
    lua_tlb->typelib->GetDocumentation(-1, NULL, NULL, &helpcontext, &helpfile);

  CHK_COM_ERR2(hr);
  
  tUtil::ShowHelp(tUtil::bstr2string(helpfile), helpcontext);

  SysFreeString(helpfile);

  return 0;
}




////////////////////////
// tLuaTypeInfo class //
////////////////////////


tLuaTypeInfo::tLuaTypeInfo(lua_State* L, ITypeInfo *p_typeinfo)
{
  method_table.Add("ShowHelp", ShowHelp, FUNC);
  method_table.Add("public", public_typeinfo, GET);
  method_table.Add("typelib", TypeLib, GET);
  method_table.Add("type", type, GET);
  method_table.Add("Load", Load, FUNC);

  filled = false;

  typeinfo = p_typeinfo;
  typeinfo->AddRef();

  helpcontext = 0;
  helpfile = NULL;

  BSTR bHelpFile;

  p_typeinfo->GetDocumentation(MEMBERID_NIL, NULL, NULL, &helpcontext, &bHelpFile);
  helpfile = tUtil::strdup(tUtil::bstr2string(bHelpFile));

  SysFreeString(bHelpFile);


  typeinfo->GetTypeAttr(&typeattr);
}

tLuaTypeInfo::~tLuaTypeInfo()
{
  typeinfo->ReleaseTypeAttr(typeattr);
  COM_RELEASE(typeinfo);

  SAFEFREE(helpfile);
}


tLuaTypeInfo* tLuaTypeInfo::pushNew(lua_State *L, ITypeInfo *p_typeinfo)
{
  CHECKPRECOND(p_typeinfo);

  // tests whether we have a cached instance
  if(pushCachedObject(L, p_typeinfo))
    return dynamic_cast<tLuaTypeInfo*>(getObject(L, -1));
    

  tLuaTypeInfo* lua_obj = NULL;

  TYPEATTR* p_typeattr = NULL;

  p_typeinfo->GetTypeAttr(&p_typeattr);

  switch(p_typeattr->typekind)
  {
  case TKIND_COCLASS:
    {
      tLuaTypeInfo::CoClass* lua_typeinfo = 
        new tLuaTypeInfo::CoClass(L, p_typeinfo);
      lua_obj = lua_typeinfo;
      
      generic_PushNew(L, lua_typeinfo, type_name, pointer_type_name);
      cacheObject(L, (void*) p_typeinfo);

      break;
    }

  case TKIND_ENUM:
    {
      tLuaTypeInfo::Enum* lua_typeinfo = 
        new tLuaTypeInfo::Enum(L, p_typeinfo);
      lua_obj = lua_typeinfo;
      
      generic_PushNew(L, lua_typeinfo, type_name, pointer_type_name);
      cacheObject(L, (void*) p_typeinfo);

      break;
    }

  case TKIND_DISPATCH:
    {
      tLuaTypeInfo::Dispinterface* lua_typeinfo = 
        new tLuaTypeInfo::Dispinterface(L, p_typeinfo);
      lua_obj = lua_typeinfo;
      
      generic_PushNew(L, lua_typeinfo, type_name, pointer_type_name);
      cacheObject(L, (void*) p_typeinfo);

      break;
    }

  default:
    {
      tLuaTypeInfo::Generic* lua_typeinfo = new tLuaTypeInfo::Generic(L, p_typeinfo);
      lua_obj = lua_typeinfo;

      generic_PushNew(L, lua_typeinfo, type_name, pointer_type_name);
      cacheObject(L, (void*) p_typeinfo);

      break;
    }
  }

  p_typeinfo->ReleaseTypeAttr(p_typeattr);

  return lua_obj;
}

void tLuaTLB::Init(lua_State* L)
{
  RegisterType(L, type_name, pointer_type_name);
  tLuaTypeInfo::Init(L);
}


void tLuaTypeInfo::Init(lua_State* L)
{
  RegisterType(L, type_name, pointer_type_name);
}

void tLuaTypeInfo::fillTable(lua_State* L)
{
  LUASTACK_SET(L);

  // Fills table with static values
  BSTR name, description;

  typeinfo->GetDocumentation(MEMBERID_NIL, &name, &description, NULL, NULL);

  lua_pushstring(L, "name");
  lua_pushstring(L, tUtil::bstr2string(name));
  lua_settable(L, -3);

  lua_pushstring(L, "description");
  lua_pushstring(L, tUtil::bstr2string(description));
  lua_settable(L, -3);

  SysFreeString(name);
  SysFreeString(description);

  LUASTACK_CLEAN(L, 0);

}


// tLuaCoClassTypeInfo: handles typeinfos of coclass elements

tLuaTypeInfo::CoClass::CoClass(lua_State* L, ITypeInfo* typeinfo) :
  tLuaTypeInfo(L, typeinfo)
{
  method_table.Add("new", New, FUNC);
}

int tLuaTypeInfo::CoClass::New(tLuaObject* lua_object, lua_State* L)
{
  LUASTACK_SET(L);
  LuaBeans *lbeans = getLuaBeans(L);

	tLuaTypeInfo::CoClass* lua_typeinfo =
    dynamic_cast<tLuaTypeInfo::CoClass*>(lua_object);

  IDispatch* pdisp = NULL;
  tLuaCOM* lcom = NULL;
  IPersistStreamInit* psi = NULL;

  try
  {
    HRESULT hr = 
      lua_typeinfo->typeinfo->CreateInstance(NULL, IID_IDispatch, (void **) &pdisp);
    CHK_COM_ERR2(hr);

    // Initializes object (some require this step to work)
    hr = pdisp->QueryInterface(IID_IPersistStreamInit, (void**) &psi);
    if(SUCCEEDED(hr))
      psi->InitNew();

    lcom = 
      tLuaCOM::CreateLuaCOM(pdisp, lua_typeinfo->typeinfo, lbeans);
  }
  catch(class tLuaCOMException& e)
  {
    luacom_error(L, e.getMessage());

    COM_RELEASE(pdisp);
    COM_RELEASE(psi);

    return 0;
  }

  COM_RELEASE(pdisp);
  COM_RELEASE(psi);

  lbeans->push(lcom);

  LUASTACK_CLEAN(L, 1);

  return 1;
}


tLuaTypeInfo::Generic::Generic(lua_State* L, ITypeInfo* typeinfo) :
  tLuaTypeInfo(L, typeinfo)
{
}


tLuaTypeInfo::Enum::Enum(lua_State *L, ITypeInfo* typeinfo) :
  tLuaTypeInfo(L, typeinfo)
{
}

void tLuaTypeInfo::Enum::fillTable(lua_State* L)
{
  LUASTACK_SET(L);

  int i = 0;
  VARDESC* pvardesc = NULL;
  HRESULT hr;
  tLuaCOMTypeHandler typehandler(NULL, getLuaBeans(L));
  BSTR name;

  lua_newtable(L);
  lua_pushstring(L, "values");
  lua_pushvalue(L, -2);
  lua_settable(L, -4);
  
  for (i=0; i < typeattr->cVars; i++)
  {
    hr = typeinfo->GetVarDesc(i, &pvardesc);
    CHK_COM_ERR2(hr);
    CHECKPRECOND(pvardesc->varkind == VAR_CONST);

    // pushes enum name
    typeinfo->GetDocumentation(pvardesc->memid, &name, NULL, NULL, NULL);
    lua_pushstring(L, tUtil::bstr2string(name));
    SysFreeString(name);

    // pushes value
    typehandler.com2lua(*pvardesc->lpvarValue);

    // sets in the table
    lua_settable(L, -3);
     
    typeinfo->ReleaseVarDesc(pvardesc);
    pvardesc = NULL;
  }

  // removes table
  lua_pop(L, 1);

  LUASTACK_CLEAN(L, 0);
  // recurses to the base type
  ((tLuaTypeInfo*) this)->fillTable(L);
}



bool tLuaTypeInfo::isPublic()
{
  return false;
}

bool tLuaTypeInfo::CoClass::isPublic()
{
  if(typeattr->wTypeFlags & TYPEFLAG_FCANCREATE
    && 
     !(typeattr->wTypeFlags &
        (TYPEFLAG_FHIDDEN | TYPEFLAG_FRESTRICTED)))
     return true;
  else
    return false;
}

bool tLuaTypeInfo::Enum::isPublic()
{
  if(!(typeattr->wTypeFlags & TYPEFLAG_FHIDDEN))
     return true;
  else
    return false;
}

int tLuaTypeInfo::ShowHelp(tLuaObject* lua_obj, lua_State* L)
{
	tLuaTypeInfo* lua_typeinfo =
    dynamic_cast<tLuaTypeInfo*>(lua_obj);
  
  CHECKPRECOND(lua_typeinfo);

  if(lua_typeinfo->helpfile != NULL)
    tUtil::ShowHelp(lua_typeinfo->helpfile, lua_typeinfo->helpcontext);

  return 0;
}

int tLuaTypeInfo::public_typeinfo(tLuaObject* lua_obj, lua_State* L)
{
  LUASTACK_SET(L);

	tLuaTypeInfo* lua_typeinfo =
    dynamic_cast<tLuaTypeInfo*>(lua_obj);
  
  CHECKPRECOND(lua_typeinfo);

  luaCompat_pushBool(L, lua_typeinfo->isPublic());

  LUASTACK_CLEAN(L, 1);

  return 1;
}

int tLuaTypeInfo::TypeLib(tLuaObject* lua_obj, lua_State* L)
{
  LUASTACK_SET(L);

	tLuaTypeInfo* lua_typeinfo =
    dynamic_cast<tLuaTypeInfo*>(lua_obj);
  
  CHECKPRECOND(lua_typeinfo);

  // stores a reference to the type library
  ITypeLib* typelib = NULL;
  unsigned int dumb = 0;

  HRESULT hr = lua_typeinfo->typeinfo->GetContainingTypeLib(&typelib, &dumb);

  if(SUCCEEDED(hr))
  {
    tLuaTLB::pushNew(L, typelib);
  }
  COM_RELEASE(typelib);

  LUASTACK_CLEAN(L, 1);

  return 1;
}


int tLuaTypeInfo::Load(tLuaObject* lua_obj, lua_State* L)
{
	tLuaTypeInfo* lua_typeinfo =
    dynamic_cast<tLuaTypeInfo*>(lua_obj);
  
  CHECKPRECOND(lua_typeinfo);

  lua_typeinfo->generic_fillTable(L);

  return 0;
}

int tLuaTypeInfo::type(tLuaObject* lua_obj, lua_State* L)
{
	tLuaTypeInfo* lua_typeinfo =
    dynamic_cast<tLuaTypeInfo*>(lua_obj);
  
  CHECKPRECOND(lua_typeinfo);

  lua_pushstring(L, 
    tCOMUtil::getPrintableTypeKind(lua_typeinfo->typeattr->typekind));

  return 1;
}



tLuaTypeInfo::Dispinterface::Dispinterface(lua_State* L, ITypeInfo* typeinfo) :
  tLuaTypeInfo(L, typeinfo)
{
}


bool tLuaTypeInfo::Dispinterface::isPublic()
{
  if(!(typeattr->wTypeFlags & TYPEFLAG_FHIDDEN))
     return true;
  else
    return false;
}


void tLuaTypeInfo::Dispinterface::fillTable(lua_State* L)
{
  LUASTACK_SET(L);

  int i = 0;

  for (i=0; i < typeattr->cFuncs; i++)
  {
    // pushes function info
    tLuaFuncDesc::pushNew(L, typeinfo, i);
    
    // sets in the table
    lua_rawseti(L, -2, i+1);
  }


  LUASTACK_CLEAN(L, 0);
  // recurses to the base type
  ((tLuaTypeInfo*) this)->fillTable(L);
}


void tLuaTypeInfo::Dispinterface::tLuaFuncDesc::pushNew(lua_State* L,
                                                        ITypeInfo* ptypeinfo,
                                                        int index)
{
  CHECKPRECOND(ptypeinfo);

  FUNCDESC* pfuncdesc = NULL;

  HRESULT hr = ptypeinfo->GetFuncDesc(index, &pfuncdesc);
  CHK_COM_ERR2(hr);

    // tests whether we have a cached instance
  if(pushCachedObject(L, pfuncdesc))
  {
    ptypeinfo->ReleaseFuncDesc(pfuncdesc);
    return;
  }

  LUASTACK_SET(L);

  tLuaFuncDesc* lua_fd = new tLuaFuncDesc(ptypeinfo, pfuncdesc);

  generic_PushNew(L, lua_fd, type_name, pointer_type_name);
  cacheObject(L, (void*) pfuncdesc);

  lua_fd->fillTable(L);

  LUASTACK_CLEAN(L, 1);

  return;
}


void tLuaTypeInfo::Dispinterface::tLuaFuncDesc::fillTable(lua_State* L)
{
  LUASTACK_SET(L);

  int i = 0;
  BSTR* names = new BSTR[pfuncdesc->cParams + 1];
  BSTR documentation;
  unsigned int retrieved = 0;

  ptypeinfo->GetDocumentation(pfuncdesc->memid, NULL, &documentation, NULL, NULL);

  lua_pushstring(L, "description");
  lua_pushstring(L, tUtil::bstr2string(documentation));
  lua_rawset(L, -3);

  SysFreeString(documentation);


  ptypeinfo->GetNames(pfuncdesc->memid, names, pfuncdesc->cParams + 1, &retrieved);

  lua_pushstring(L, "name");
  lua_pushstring(L, tUtil::bstr2string(names[0]));
  lua_rawset(L, -3);

  SysFreeString(names[0]);

  lua_pushstring(L, "type");

  switch(pfuncdesc->invkind)
  {
  case INVOKE_FUNC:
    lua_pushstring(L, "method");
    break;

  case INVOKE_PROPERTYGET:
    lua_pushstring(L, "propget");
    break;

  case INVOKE_PROPERTYPUT:
  case INVOKE_PROPERTYPUTREF:
    lua_pushstring(L, "propput");
    break;
  }

  lua_rawset(L, -3);

  // stores the function parameters names
  for(i = 1; i < pfuncdesc->cParams; i++)
  {
    lua_newtable(L);

    if(i < retrieved)
    {
      lua_pushstring(L, "name");
      lua_pushstring(L, tUtil::bstr2string(names[i]));
      lua_rawset(L, -3);
      SysFreeString(names[i]);
    }

    lua_rawseti(L, -2, i);
  }

  delete names;

  LUASTACK_CLEAN(L, 0);
}

tLuaTypeInfo::Dispinterface::tLuaFuncDesc::tLuaFuncDesc(ITypeInfo* ptypeinfo_p,
                                                        FUNCDESC* pfuncdesc_p)
{
  ptypeinfo = ptypeinfo_p;
  ptypeinfo->AddRef();

  pfuncdesc = pfuncdesc_p;
}


tLuaTypeInfo::Dispinterface::tLuaFuncDesc::~tLuaFuncDesc()
{
  ptypeinfo->ReleaseFuncDesc(pfuncdesc);
  COM_RELEASE(ptypeinfo);
}

void tLuaTypeInfo::generic_fillTable(lua_State *L)
{
  fillTable(L);
}

void tLuaTypeInfo::Dispinterface::generic_fillTable(lua_State *L)
{
  fillTable(L);
}

int tLuaTypeInfo::index(lua_State *L)
{
  if(!filled)
  {
    lua_pushvalue(L, -2);
    generic_fillTable(L);
    lua_pop(L, 1);
    filled = true;
  }

  lua_rawget(L, -2);

  if(!lua_isnil(L, -1))
    return 1;
  else
    return 0;
}

void tLuaTypeInfo::Enum::generic_fillTable(lua_State *L)
{
  fillTable(L);
}
