/*
 * LUACOM.cpp
 *
 *   Bind de Lua com COM/OLE
 *
 *  Renato Cerqueira
 *  Vinicius Almendra
 *
 */

// RCS Info
static char *rcsid = "$Id: luacom.cpp,v 1.24 2002/09/25 16:42:31 almendra Exp $";
static char *rcsname = "$Name:  $";
static char *g_version = "1.0 (BETA)";

#include <string.h>
#include <iostream.h>
#include <stdlib.h>

#include <ole2.h>
#include <ocidl.h>

#include <assert.h>
#include <stdio.h>
#include <math.h>

extern "C"
{
#include "lua.h"
#include "lauxlib.h"
#include "luadebug.h"
}

#include "luabeans.h"
#include "luacom.h"
#include "tLuaDispatch.h"
#include "tLuaCOM.h"
#include "tLuaCOMException.h"
#include "tCOMUtil.h"
#include "tLuaCOMClassFactory.h"
#include "tLuaCOMTypeHandler.h"

#include "LuaAux.h"
#include "tUtil.h"


////////////////////////
//                    //
//  FUNCOES INTERNAS  //
//                    //
////////////////////////

/*
 * getLuaBeans
 *
 *  Retorna o objeto LuaBeans associado ao estado Lua fornecido 
 *
 */

LuaBeans *getLuaBeans(lua_State *L)
{
  return LuaBeans::getBeans(L, tLuaCOM::TAG);
}

static tLuaCOM* ImplInterface(ITypeLib* typelib,
                              ITypeInfo* coclasstypeinfo,
                              const char* interface_name,
                              int unlocked_ref,
                              LuaBeans* lbeans)
{
  CHECKPARAM(typelib && interface_name && lbeans);

  ITypeInfo* interface_typeinfo = 
    tCOMUtil::GetInterfaceTypeInfo(typelib, interface_name);

  if(interface_typeinfo == NULL)
    return NULL;

  tLuaDispatch* iluacom = 
    tLuaDispatch::CreateLuaDispatch(interface_typeinfo, unlocked_ref, lbeans);

  if(iluacom == NULL)
  {
    interface_typeinfo->Release();
    return NULL;
  }

  tLuaCOM *lcom = tLuaCOM::CreateLuaCOM(iluacom, coclasstypeinfo, lbeans);

  interface_typeinfo->Release();
  iluacom->Release();

  return lcom;
}


///////////////////////////////////
//                               //
//  FUNCOES EXPORTADAS PARA LUA  //
//                               //
///////////////////////////////////

/*
 * lc_ShowHelp
 *
 * Parametros: 
 *
 *  (1) objeto luacom
 *
 *   Mostra Help da type library associada ao objeto
 *   luacom
 */

static int lc_ShowHelp(lua_State *L)
{
  char *pHelpFile = NULL; 
  unsigned long context = 0;

  tLuaCOM* luacom = (tLuaCOM *) getLuaBeans(L)->check_tag(1);

  luacom->getHelpInfo(&pHelpFile, &context);

  if(pHelpFile != NULL)
  {
    if(context != 0)
      WinHelp(NULL, pHelpFile, HELP_CONTEXT, context);
    else
      WinHelp(NULL, pHelpFile, HELP_FINDER, 0);
  }

  return 0;
}

/*
 * lc_Connect
 *
 *   Parametros (Lua):
 *
 *    objeto luacom
 *    tabela que implementara a interface de conexao
 *
 *   Retorno
 *    objeto luacom que encapsula objeto luacom implementado pela
 *    tabela fornecida ou nil se nao for possvel estabelecer a
 *    conexao
 *
 *   Cria um connection point utilizando a tabela
 *   passada como parametro e o conecta com o objeto
 *   passado tambem como parametro
 */

static int lc_Connect(lua_State *L)
{
  tLuaDispatch *server_disp = NULL;
  ITypeInfo *pTypeinfo = NULL;
  int ref = 0;
  LuaBeans *lbeans = getLuaBeans(L);

  // check parameters
  tLuaCOM* client = (tLuaCOM *) lbeans->check_tag(1);

  if(lua_type(L, 2) != LUA_TTABLE && lua_type(L, 2) != LUA_TUSERDATA)
  {
    luaL_argerror(L, 2, "Implementation must be a table or a userdata");
  }

  /* gets a unlocked reference to the implementation */
  lua_pushvalue(L, 2);
  int unlocked_ref = lua_ref(L, 0);

  pTypeinfo = client->GetDefaultEventsInterface();

  if(pTypeinfo == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  server_disp = 
    tLuaDispatch::CreateLuaDispatch(pTypeinfo, unlocked_ref, lbeans);

  if(server_disp == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  tLuaCOM* server =
    tLuaCOM::CreateLuaCOM(server_disp, client->GetCoClassTypeInfo(), lbeans);

  COM_RELEASE(server_disp);

  if(server == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  client->addConnection(server);

  lbeans->push(server); 

  return 1;
}

/*
 *  lc_ImplInterfaceFromTypelib (Lua2C)
 *
 *  Parametros:
 *    1. Tabela de implementacao
 *    2. Nome da type library
 *    3. Nome da interface a ser implementada
 *    4. (Opcional) Nome da CoClass da qual a interface faz parte
 *       necessario no caso de se querer expor objetos implementados
 *       com essa interface ou no caso de se querer localizar uma
 *       interface source (eventos)
 *
 *  Retorno:
 *    1. Objeto LuaCOM que encapsula a implementacao via Lua
 *        ou nil em caso de erro
 *
 *  Implementa uma interface descrida em uma type library
 *  dada
 */

static int lc_ImplInterfaceFromTypelib(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  if(lua_type(L, 1) != LUA_TTABLE && lua_type(L,1) != LUA_TUSERDATA)
  {
    luaL_argerror(L, 1, "Implementation must be a table or a userdata");
  }

  lua_pushvalue(L, 1);
  const int unlocked_ref = lua_ref(L, 0);

  const char* typelib_name = luaL_check_lstr(L, 2, NULL);
  const char* pcInterface = luaL_check_lstr(L, 3, NULL);
  const char* coclassname = luaL_opt_lstr(L, 4, NULL, NULL);

  // gets typelib
  ITypeLib* typelib = tCOMUtil::LoadTypeLibByName(typelib_name);

  if(!typelib)
  {
    lua_pushnil(L);
    return 1;
  }

  // gets coclass typeinfo
  ITypeInfo *coclassinfo = NULL;

  if(coclassname)
  {
    coclassinfo = tCOMUtil::GetCoClassTypeInfo(typelib, coclassname);

    if(!coclassinfo)
    {
      typelib->Release();
      lua_pushnil(L);
      return 1;
    }
  }

  tLuaCOM* lcom = 
    ImplInterface(typelib, coclassinfo, pcInterface, unlocked_ref, lbeans);

  if(coclassinfo != NULL)
    COM_RELEASE(coclassinfo);

  COM_RELEASE(typelib);

  if(!lcom)
  {
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom); 
  return 1;
}

/* 
 *  lc_ImplInterface (Lua2C)
 *
 *  Cria uma implementacao IDispatch para um
 *  objeto Lua dado um ProgID e o nome da
 *  interface
 *
 *  In: Implementation: table,
 *      ProgID: string,
 *      Interface: string,
 *
 *  Out: LuaCOM_obj (table with tag LuaCOM)
*/

static int lc_ImplInterface(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  // gets parameters
  if(lua_type(L, 1) != LUA_TTABLE && lua_type(L,1) != LUA_TUSERDATA)
  {
    luaL_argerror(L, 1, "Implementation must be a table or a userdata");
  }

  // pushes lua table on top of stack
  lua_pushvalue(L, 1);
  const int unlocked_ref = lua_ref(L, 0);

  const char* pcProgID = luaL_check_lstr(L, 2, NULL);
  const char* pcInterface = luaL_check_lstr(L, 3, NULL);

  // gets typelib
  CLSID clsid;

  if(tCOMUtil::ProgID2CLSID(&clsid, pcProgID) != S_OK)
  {
    lua_pushnil(L);
    return 1;
  }

  ITypeLib* typelib = tCOMUtil::LoadTypeLibFromCLSID(clsid);

  if(typelib == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  // gets coclass typeinfo (we don't check return value
  // because NULL is a valid value

  ITypeInfo* coclassinfo = tCOMUtil::GetCoClassTypeInfo(typelib, clsid);

  tLuaCOM* lcom =
    ImplInterface(typelib, coclassinfo, pcInterface, unlocked_ref, lbeans);

  COM_RELEASE(coclassinfo);
  COM_RELEASE(typelib);

  if(lcom == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom);

  return 1;
}

/*
 *  lc_CLSIDfromProgID
 *  Retorna string contendo CLSID associado a um ProgID
 */

static int lc_CLSIDfromProgID(lua_State *L)
{
   const char* str = luaL_check_lstr(L, 1, NULL);
   wchar_t* clsid_str;
   wchar_t* progId;
   CLSID clsid;
   HRESULT hr;

   progId = (wchar_t*) malloc( (strlen(str) + 1) * sizeof(wchar_t));
   mbstowcs(progId,str,strlen(str)+1);

   hr = CLSIDFromProgID(progId, &clsid);
   free(progId);
   if (FAILED(hr))
   {
      cout << "Error looking for class " << str << endl;
      lua_pushnil(L);
      return 1;
   }
   hr = StringFromCLSID(clsid, &clsid_str);
   if (FAILED(hr))
   {
      cout << "Error looking for prog_id" << endl;
      lua_pushnil(L);
      return 1;
   }

   char* id_str = (char*) malloc( (wcslen(clsid_str) + 1) * sizeof(char));
   wcstombs(id_str,clsid_str,wcslen(clsid_str)+1);
   lua_pushstring(L, id_str);
   free(id_str);

   return 1;
}

/*
 *  lc_ProgIDfromCLSID
 *  Retorna string contendo o ProgID associado a um CLSID
 */

static int lc_ProgIDfromCLSID(lua_State *L)
{
   const char* str = luaL_check_lstr(L, 1, NULL);
   wchar_t* clsid_str;
   LPOLESTR progId;
   CLSID clsid;
   HRESULT hr;

   clsid_str = (wchar_t*) malloc( (strlen(str) + 1) * sizeof(wchar_t));
   mbstowcs(clsid_str,str,strlen(str)+1);

   hr = CLSIDFromString(clsid_str, &clsid);
   free(clsid_str);
   if (FAILED(hr))
   {
      cout << "Error looking for class " << str << endl;
      lua_pushnil(L);
      return 1;
   }
   hr = ProgIDFromCLSID(clsid, &progId);
   if (FAILED(hr))
   {
      cout << "Error looking for prog_id"<< endl;
      lua_pushnil(L);
      return 1;
   }

   char* id_str = (char*) malloc( (wcslen(progId) + 1) * sizeof(char));
   wcstombs(id_str,progId,wcslen(progId)+1);
   lua_pushstring(L, id_str);
   free(id_str);

   return 1;
}

/*
 *  lc_CreateObject
 *  Retorna um objeto LuaCOM que instancia o objeto
 *  COM identificado pelo ProgID dado
 */

static int lc_CreateObject(lua_State *L)
{
  HRESULT hr       = S_OK;
  LPDISPATCH pdisp = NULL;
  LuaBeans *lbeans = getLuaBeans(L);

  const char *progId = luaL_check_lstr(L, 1, NULL);

  CLSID clsid;
  hr = tCOMUtil::ProgID2CLSID(&clsid, progId);

  if (FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  hr = CoCreateInstance(clsid,NULL,CLSCTX_ALL, IID_IDispatch,(void**)&pdisp);

  if (FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  // gets coclasstypeinfo
  ITypeInfo* coclassinfo = tCOMUtil::GetCoClassTypeInfo(pdisp, clsid);

  // we don't check coclassinfo, because NULL is a valid value

  tLuaCOM* lcom = tLuaCOM::CreateLuaCOM(pdisp, coclassinfo, lbeans);

  COM_RELEASE(pdisp);
  COM_RELEASE(coclassinfo);

  if(lcom == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom);

  return 1;
}


/*
 *  lc_GetObject
 *  Cria um objeto LuaCOM associado a uma instancia
 *  ja' existente do objeto COM identificado pelo 
 *  ProgID dado
 */

static int lc_GetObject(lua_State *L)
{
  HRESULT hr       = S_OK;
  IDispatch* pdisp = NULL;
  IUnknown* punk = NULL;
  LuaBeans *lbeans = getLuaBeans(L);

  const char *progId = luaL_check_lstr(L, 1, NULL);

  CLSID clsid;
  hr = tCOMUtil::ProgID2CLSID(&clsid, progId);

  if (FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  hr = GetActiveObject(clsid,NULL,&punk);

  if (FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  hr = punk->QueryInterface(IID_IDispatch, (void **) &pdisp);

  if(FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  punk->Release();
  punk = NULL;

  // gets coclasstypeinfo
  ITypeInfo* coclassinfo = tCOMUtil::GetCoClassTypeInfo(pdisp, clsid);

  tLuaCOM* lcom = tLuaCOM::CreateLuaCOM(pdisp, coclassinfo, lbeans);

  COM_RELEASE(pdisp);
  COM_RELEASE(coclassinfo);

  if(lcom == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom);

  return 1;
}

/*
 *  lc_addConnection
 *  Associa um connection point de um objeto COM com
 *  outro objeto COM
 */

static int lc_addConnection(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  tLuaCOM* client = (tLuaCOM*) lbeans->check_tag(1);
  tLuaCOM* server = (tLuaCOM*) lbeans->check_tag(2);

  bool result = client->addConnection(server);

  if(result == false)
    lua_pushnil(L);
  else
    lua_pushnumber(L, 1);

  return 1;
}

/*
 *  lc_releaseConnection
 *  Desfaz connection point associado a um objeto LuaCOM
 */

static int lc_releaseConnection(lua_State *L)
{
  tLuaCOM* lcom = (tLuaCOM*) getLuaBeans(L)->check_tag(1);

  lcom->releaseConnection();

  return 0;
}

//
// lc_isMember
//
//  Informa se existe algum metodo ou propriedade com
//  o nome passado como parametro em lua
//

static int lc_isMember(lua_State *L)
{
  // objeto luacom
  tLuaCOM* lcom = (tLuaCOM*) getLuaBeans(L)->check_tag(1);
  const char* member_name = luaL_check_lstr(L, 2, NULL);

  if(lcom->isMember(member_name))
    lua_pushnumber(L, 1);
  else
    lua_pushnil(L);

  return 1;
}

/*
 * lc_NewObject
 *
 *  Cria um objeto COM implementado via uma tabela Lua.
 *
 */

static int lc_NewObject(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  // gets parameters

  if(lua_type(L, 1) != LUA_TTABLE && lua_type(L,1) != LUA_TUSERDATA)
  {
    luaL_argerror(L, 1, "Implementation must be a table or a userdata");
  }

  // pushes lua table on top of stack
  lua_pushvalue(L, 1);
  const int unlocked_ref = lua_ref(L, 0);

  const char* pcProgID = luaL_check_lstr(L, 2, NULL);

  // gets typelib
  CLSID clsid;

  if(tCOMUtil::ProgID2CLSID(&clsid, pcProgID) != S_OK)
  {
    lua_pushnil(L);
    return 1;
  }

  ITypeLib* typelib = tCOMUtil::LoadTypeLibFromCLSID(clsid);

  if(typelib == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  // gets coclass typeinfo

  ITypeInfo* coclassinfo = tCOMUtil::GetCoClassTypeInfo(typelib, clsid);

  if(coclassinfo == NULL)
  {
    typelib->Release();
    lua_pushnil(L);
    return 1;
  }

  // gets the default interface typeinfo
  ITypeInfo* interface_typeinfo =
    tCOMUtil::GetDefaultInterfaceTypeInfo(coclassinfo, false);

  if(interface_typeinfo == NULL)
  {
    coclassinfo->Release();
    typelib->Release();

    lua_pushnil(L);
    return 1;
  }

  tLuaDispatch* iluacom = 
    tLuaDispatch::CreateLuaDispatch(interface_typeinfo, unlocked_ref, lbeans);

  tLuaCOM *lcom = NULL;

  if(iluacom != NULL)
  {
    lcom = tLuaCOM::CreateLuaCOM(iluacom, coclassinfo, lbeans);
  }

  COM_RELEASE(iluacom);
  COM_RELEASE(interface_typeinfo);
  COM_RELEASE(coclassinfo);
  COM_RELEASE(typelib);

  if(lcom == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom);

  return 1;
}


/*
 *  luacom_ExposeObject
 *
 *    Creates a class factory that exposes a
 *    COM object
 *
 *  Parameters:
 *
 *    1. LuaCOM object
 *
 *  Return values
 *
 *    1. Cookie to unexpose object
 *
 */

static int lc_ExposeObject(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  // check parameters
  tLuaCOM* luacom = (tLuaCOM *) lbeans->check_tag(1);

  tLuaCOMClassFactory* luacom_cf = 
      new tLuaCOMClassFactory(luacom->GetIDispatch());

  ITypeInfo *coclassinfo = luacom->GetCoClassTypeInfo();
  
  CLSID clsid;
  HRESULT hr = tCOMUtil::GetCLSID(coclassinfo, &clsid);

  DWORD cookie = -1;

  hr = CoRegisterClassObject(
    clsid,
    luacom_cf,
    CLSCTX_LOCAL_SERVER,
    REGCLS_SINGLEUSE,
    &cookie);

  if(FAILED(hr))
  {
    delete luacom_cf;
    lua_pushnil(L);
    return 1;
  }

  lua_pushnumber(L, cookie);

  return 1;
}


/*
 *  luacom_RevokeObject
 *
 *    Revokes a previously registered class factory
 *
 *  Parameters:
 *
 *    1. Cookie
 *
 *  Return values
 *
 *    1. non-nil if succeeded
 *
 */

static int lc_RevokeObject(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  // check parameters
  const int cookie = (int) luaL_check_number(L, 1);

  // revokes class object
  HRESULT hr = CoRevokeClassObject(cookie);

  if(FAILED(hr))
  {
    lua_pushnil(L);
  }
  else
  {
    lua_pushnumber(L, 1);
  }

  return 1;
}


/*
 *  luacom_RegisterObject
 *
 *    Registers a COM Object in the system registry
 *
 *  Parameters:
 *
 *    1. registration table or userdata
 *
 *  Return values
 *
 *    1. non-nil if successful
 *
 */


static int lc_RegisterObject(lua_State *L)
{
  HRESULT hr = S_OK;
  const int bufsize = 1000;

  if(lua_type(L, 1) != LUA_TTABLE && lua_type(L,1) != LUA_TUSERDATA)
  {
    luaL_argerror(L, 1, "Registration info must be a table or userdata");
  }

  // gets the registration information from the registration table
  lua_pushstring(L, "VersionIndependentProgID");
  lua_gettable(L, 1);
  const char* VersionIndependentProgID = lua_tostring(L, -1);

  // gets the registration information from the registration table
  lua_pushstring(L, "ProgID");
  lua_gettable(L, 1);
  const char* ProgID = lua_tostring(L, -1);

  lua_pushstring(L, "TypeLib");
  lua_gettable(L, 1);
  const char* typelib_path = lua_tostring(L, -1);

  lua_pushstring(L, "CoClass");
  lua_gettable(L, 1);
  const char* CoClass = lua_tostring(L, -1);

  lua_pushstring(L, "ComponentName");
  lua_gettable(L, 1);
  const char* ComponentName= lua_tostring(L, -1);

  lua_pushstring(L, "Arguments");
  lua_gettable(L, 1);
  const char* arguments = lua_tostring(L, -1);

  if(!
    (ProgID &&
     typelib_path &&
     CoClass))
  {
    lua_pushnil(L);
    return 1;
  }

  // Loads and registers the typelib
  ITypeLib *typelib = NULL;

  {
    wchar_t wcTypelib_path[bufsize];
    
    mbstowcs(wcTypelib_path, typelib_path, strlen(typelib_path)+1);

    hr = LoadTypeLibEx(wcTypelib_path, REGKIND_REGISTER, &typelib);
  }

  if(FAILED(hr))
  {
    lua_pushnil(L);
    return 1;
  }

  // Gets the type library version and LIBID

  char version[30];
  char libId[bufsize];

  {
    TLIBATTR *plibattr = NULL;

    typelib->GetLibAttr(&plibattr);

    // gets version
    sprintf(version, "%d.%d", plibattr->wMajorVerNum, plibattr->wMinorVerNum);

    // gets libid
    wchar_t *wcLibId = NULL;
    StringFromCLSID(plibattr->guid, &wcLibId);
    wcstombs(libId, wcLibId, wcslen(wcLibId)+1);
    CoTaskMemFree(wcLibId);

    typelib->ReleaseTLibAttr(plibattr);
  }

  // gets the CoClass TypeInfo to get the CLSID
  ITypeInfo *coclassinfo = tCOMUtil::GetCoClassTypeInfo(typelib, CoClass);

  if(coclassinfo == NULL)
  {
    typelib->Release();
    lua_pushnil(L);
    return 1;
  }

  // gets the CLSID
  
  char clsid[bufsize];
  
  {
    TYPEATTR* ptypeattr = NULL;
    wchar_t* wcClsid=  NULL;

    coclassinfo->GetTypeAttr(&ptypeattr);

    StringFromCLSID(ptypeattr->guid, &wcClsid);
    wcstombs(clsid, wcClsid,wcslen(wcClsid)+1);

    coclassinfo->ReleaseTypeAttr(ptypeattr);
    CoTaskMemFree(wcClsid);
  }

  COM_RELEASE(coclassinfo);
  COM_RELEASE(typelib);

  //// Now we have all the information needed to perform the registration

  // registers ProgID
  char ID[bufsize];
  char CLSID[bufsize];
  char ModulePath[bufsize];

  // Be safe with null strings in these stack-allocated strings.
  ID[0] = 0;
  CLSID[0] = 0;
  ModulePath[0] = 0;

  GetModuleFileName(
    NULL,
    ModulePath,
    sizeof(ModulePath));

  if(arguments)
  {
    strcat(ModulePath, " ");
    strcat(ModulePath, arguments);
  }

  // Create some base key strings.
  strcpy(CLSID, "CLSID\\");
  strcat(CLSID, clsid);

  // Create ProgID keys.
  tCOMUtil::SetRegKeyValue(
    ProgID,
    NULL,
    ComponentName);

  tCOMUtil::SetRegKeyValue(
    ProgID,
    "CLSID",
    clsid);

  // Create VersionIndependentProgID keys.
  tCOMUtil::SetRegKeyValue(
    VersionIndependentProgID,
    NULL,
    ComponentName);

  tCOMUtil::SetRegKeyValue(
    VersionIndependentProgID,
    "CurVer",
    ProgID);

  tCOMUtil::SetRegKeyValue(
    VersionIndependentProgID,
    "CLSID",
    clsid);

  // Create entries under CLSID.
  tCOMUtil::SetRegKeyValue(
    CLSID,
    NULL,
    ComponentName);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "ProgID",
    ProgID);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "VersionIndependentProgID",
    VersionIndependentProgID);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "LocalServer32",
    ModulePath);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "Version",
    version);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "TypeLib",
    libId);

  tCOMUtil::SetRegKeyValue(
    CLSID,
    "Programmable",
    NULL);



  // signals success
  lua_pushnumber(L, 1);
  return 1;
}



/*
 *  luacom_GetIUnknown
 *
 *    Returns the a IUnknown interface for a
 *    LuaCOM object
 *
 *  Parameters:
 *
 *    1. LuaCOM object
 *
 *  Return values
 *
 *    1. IUnknown pointer (a userdata)
 *
 */


static int lc_GetIUnknown(lua_State *L)
{
  LuaBeans *lbeans = getLuaBeans(L);

  // check parameters
  tLuaCOM* luacom = (tLuaCOM *) lbeans->check_tag(1);

  IDispatch* pdisp = luacom->GetIDispatch();
  IUnknown* punk = NULL;
  
  pdisp->QueryInterface(IID_IUnknown, (void **) &punk);

  // checks whether there is a usertag for this IUnknown
  // If exists, simply uses it

  // gets IUnknown tag
  lua_getglobal(L, LUACOM_IUNKNOWN_TAGNAME);
  const int IUnknown_tag = (int) lua_tonumber(L, -1);
  lua_remove(L, -1);

  // pushes userdata
  lua_pushusertag(L, punk, LUA_ANYTAG);

  if(lua_tag(L, -1) == IUnknown_tag)
  {
    // there was already a userdata for that pointer, 
    // so we have to decrement the reference count,
    // as the garbage collector will be called just
    // once for any userdata
    punk->Release();
  }
  else
  {
    // this is the first userdata with this value,
    // so corrects the tag
    lua_settag(L, IUnknown_tag);
  }

  return 1;
}


//////////////////////////////////////
//                                  //
//  TAG METHODS DO USERTAG IUNKNOWN //
//                                  //
//////////////////////////////////////

/*
 * tag method que gerencia garbage collection
 */

static int IUnknown_tag_gc(lua_State *L)
{
   IUnknown* punk = (IUnknown*) lua_touserdata(L, 1);

   if(punk != NULL)
     punk->Release();

   return 0;
}


////////////////////////////////////
//                                //
//  TAG METHODS DO OBJETO LUACOM  //
//                                //
////////////////////////////////////

/*
 * tag method que gerencia garbage collection
 */

static int gc_fb(lua_State *L)
{
   tLuaCOM* lcom = (tLuaCOM*) lua_touserdata(L, 1);

   assert(lcom);

   if(lcom != NULL)
     delete lcom;

   return 0;
}

/*
 * tag method que gerencia atribuicoes a campos
 * do objeto luacom
 */

static int settable_fb(lua_State *L)
{
   /* indexes to the parameters coming from lua */
   const int table = 1; 
   const int index = 2;
   const int val = 3;

   LuaBeans *lbeans = getLuaBeans(L);

   try
   {
     tLuaCOM* lcom = (tLuaCOM*) lbeans->from_lua(table);

     if (lua_isstring(L, index) && lcom != NULL)
     {
        const char* name = lua_tostring(L, index);
        DISPID dispid;
        HRESULT hr = lcom->get_dispid(name, &dispid);

        if (!FAILED(hr))
        {
           FUNCDESC* pfuncdescr = NULL;

           hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYPUT,&pfuncdescr );

           if(FAILED(hr))
             hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYPUTREF,&pfuncdescr );

           if ( !FAILED(hr) )
           {
             int result = lcom->putprop(pfuncdescr,val);
             lcom->ReleaseFuncDesc(pfuncdescr);

             // se chamada foi bem-sucedida, retorna
             return result;
           }
        }
     }
   }
   catch(class tLuaCOMException& e)
   {
    switch(e.code)
    {
    case tLuaCOMException::INTERNAL_ERROR:
    case tLuaCOMException::PARAMETER_OUT_OF_RANGE:
    case tLuaCOMException::UNSUPPORTED_FEATURE:
    default:
      lua_error(L, e.getMessage());
      break;
    }
   }

   lua_rawset(L, table);

   return 0;
}


/*
 * Closure que gerencia chamadas de metodos
 *
 * In: userdata (tLuaCOM *), userdata (FUNCDESC *),
 *     table | userdata, ...
 */
static int lc_call(lua_State *L)
{
  tLuaCOM* lcom       = NULL;
  HRESULT hr          = S_OK;
  DISPID dispid       = -1;
  FUNCDESC *pFuncDesc = NULL;
  LuaBeans *lbeans = getLuaBeans(L);
  int num_return_values = -1;

  // upvalues
  const int luacom_obj_param  = -2;
  const int dispid_param      = -1;

  // numero de parametros passados
  const int num_params = lua_gettop(L) - 2;

  try
  {
    // obtem objeto luacom
    lcom = (tLuaCOM *) lua_touserdata(L, luacom_obj_param);

    CHECK(lcom, INTERNAL_ERROR);

    // pega o dispid
    dispid = (DISPID) lua_tonumber(L, dispid_param);

    // Testa para o caso de ser metodo
    hr = lcom->get_funcdescr(dispid,INVOKE_FUNC,&pFuncDesc);

    if (SUCCEEDED(hr))  //e' metodo
    {
      // Forca usuario a usar : na chamada a metodo, o que
      // faz com que lua passe o argumento "self". Esse argumento
      // nao e' utilizado na implementacao, mas e' exigido para
      // garantir uniformidade sintatica

      // checks if, at least, the 'self' parameter was passed
      if(num_params == 0)
      {
        lua_error(L, "Method call without ':'");
        return 0;
      }

      const int first_param = 1;

      // sets the parameter list excluding the 'self' param
      tLuaObjList& params =tLuaObjList(first_param+1, num_params - 1);

      num_return_values = lcom->call(pFuncDesc, params);

      lcom->ReleaseFuncDesc(pFuncDesc);

      return num_return_values;
    }    

    //// tenta para o caso de ser propget ou propput parametrizado



    // testa o propget

    hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYGET,&pFuncDesc);    

    if(SUCCEEDED(hr) && pFuncDesc->cParams == num_params)
    {
      // e' propget
      tLuaObjList &params = tLuaObjList(1, num_params);

      num_return_values = lcom->call(pFuncDesc, params);

      lcom->ReleaseFuncDesc(pFuncDesc);

      return num_return_values;
    }

    // tenta o propput (ou propputref) parametrizados

    hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYPUT,&pFuncDesc);

    if(FAILED(hr)) // tenta propputref
      hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYPUTREF,&pFuncDesc);

    if(SUCCEEDED(hr) && pFuncDesc->cParams == num_params) 
    {
      tLuaObjList &params = tLuaObjList(1, num_params);
      num_return_values = lcom->call(pFuncDesc, params);

      lcom->ReleaseFuncDesc(pFuncDesc);

      return num_return_values;
    }

  }
  catch(class tLuaCOMException& e)
  {
    switch(e.code)
    {
    case tLuaCOMException::INTERNAL_ERROR:
    case tLuaCOMException::PARAMETER_OUT_OF_RANGE:
    case tLuaCOMException::UNSUPPORTED_FEATURE:
    default:
      lua_error(L, e.getMessage());
      break;
    }
  }

  // nao conseguiu chamar nada, devolve nada!

  return 0;
}


/*
 * tag method que gerencia leituras de campos
 * do objeto luacom
 *
 * In: table, index
 */

static int index_fb(lua_State *L)
{
  // variaveis utilizadas
  DISPID dispid;
  HRESULT hr              = S_OK;
  FUNCDESC* pfuncdescr    = NULL;
  tLuaCOM* lcom           = NULL;
  LuaBeans *lbeans = getLuaBeans(L);

  try
  {
    lcom = (tLuaCOM*) lbeans->from_lua(1);
    CHECK(lcom, INTERNAL_ERROR);

    // obtem campo acessado pelo programa
    {
      const int index = 2;

      CHECK(lua_isstring(L, index), INTERNAL_ERROR);

      // tenta obter dispid correspondente ao nome fornecido
      const char *name = lua_tostring(L, index);
      hr = lcom->get_dispid(name, &dispid);

      // se nome fornecido nao corresponder a nenhum dispid...
      if (FAILED(hr)) 
        return 0;
    }

    // se for propget sem parametro, executa direto

    hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYGET,&pfuncdescr);

    if (SUCCEEDED(hr) && pfuncdescr->cParams == 0)  
    {
      //e' propget sem parametros. Executa direto

      lcom->getprop(pfuncdescr);
    }
    else // repassa para lc_call decidir o que deve ser chamado
    {
      lua_pushusertag(L, (void *) lcom, LUA_ANYTAG);
      lua_pushnumber(L, dispid);
      lua_pushcclosure(L, lc_call, 2);
    }

    lcom->ReleaseFuncDesc(pfuncdescr);
  }
  catch(class tLuaCOMException& e)
  {
    switch(e.code)
    {
    case tLuaCOMException::INTERNAL_ERROR:
    case tLuaCOMException::PARAMETER_OUT_OF_RANGE:
    case tLuaCOMException::UNSUPPORTED_FEATURE:
    default:
      lua_error(L, e.getMessage());
      break;
    }

  }

  return 1;
}


/////////////////////////////////////////////
//                                         //
//  TABELA DAS FUNCOES EXPORTADAS PARA LUA //
//                                         //
/////////////////////////////////////////////

static struct luaL_reg functions_tb []= 
{
  {"luacom_CreateObject",lc_CreateObject},
  {"luacom_GetObject",lc_GetObject},
  {"luacom_CLSIDfromProgID",lc_CLSIDfromProgID},
  {"luacom_ImplInterface",lc_ImplInterface},
  {"luacom_ImplInterfaceFromTypelib",lc_ImplInterfaceFromTypelib},
  {"luacom_addConnection", lc_addConnection},
  {"luacom_releaseConnection", lc_releaseConnection},
  {"luacom_ProgIDfromCLSID",lc_ProgIDfromCLSID},
  {"luacom_isMember",lc_isMember},
  {"luacom_Connect",lc_Connect},
  {"luacom_ShowHelp",lc_ShowHelp},
  {"luacom_NewObject", lc_NewObject},
  {"luacom_ExposeObject", lc_ExposeObject},
  {"luacom_RevokeObject", lc_RevokeObject},
  {"luacom_RegisterObject", lc_RegisterObject},
  {"luacom_GetIUnknown", lc_GetIUnknown}
};
  
static struct luaL_reg tagmethods_tb[]= {{"index", index_fb},
                                  {"settable",settable_fb}};
static struct luaL_reg ud_tagmethods_tb[]= {{"gc", gc_fb}};


/////////////////////////////////////
//                                 //
//  FUNCOES EXPORTADAS PARA C/C++  //
//                                 //
/////////////////////////////////////

/*
 *  luacom_IDispatch2LuaCOM
 *  Recebe um ponteiro void* que deve ser
 *  um ponteiro para uma implementacao de uma
 *  interface IDispatch. E' criado um objeto
 *  LuaCOM cuja implementacao e' dada por essa
 *  interface. O objeto criado e' colocado na
 *  pilha de Lua
 */

LUACOM_API int luacom_IDispatch2LuaCOM(lua_State *L, void *pdisp_arg)
{
  LuaBeans *lbeans = getLuaBeans(L);

  if(pdisp_arg == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  tLuaCOM* lcom = tLuaCOM::CreateLuaCOM((IDispatch*)pdisp_arg, NULL, lbeans);

  if(lcom == NULL)
  {
    lua_pushnil(L);
    return 1;
  }

  lbeans->push(lcom);

  return 1;
}

/*
 *  Inicializacao da biblioteca
 */

LUACOM_API void luacom_open(lua_State *L)
{
  if(L == NULL)
    return;

  luaL_openlib(L, functions_tb,sizeof(functions_tb)/sizeof(struct luaL_reg));

  LuaBeans *lbeans = LuaBeans::createBeans(L, tLuaCOM::TAG);
  lbeans->reg_tagmethods(tagmethods_tb,
                        sizeof(tagmethods_tb)/sizeof(struct luaL_reg));
  lbeans->reg_ud_tagmethods(ud_tagmethods_tb,
                          sizeof(ud_tagmethods_tb)/sizeof(struct luaL_reg));


  // Creates a lua tag for IUnknown pointes
  int IUnknown_tag = lua_newtag(L);

  // Sets a tag method for garbage collection
  lua_pushcfunction(L, IUnknown_tag_gc);
  lua_settagmethod(L, IUnknown_tag, "gc");

  // stores in the table
  lua_pushnumber(L, IUnknown_tag);
  lua_setglobal(L, LUACOM_IUNKNOWN_TAGNAME);
}

/*
 *  Fechamento da biblioteca
 */

LUACOM_API void luacom_close(lua_State *L)
{
  // does nothing (at least now)
}







