/*
 * tLuaCOM.cpp
 *
 *  Implementacao da classe tLuaCOM
 *
 * Renato Cerqueira
 * Vinicius Almendra
 */

// RCS Info
static char *rcsid = "$Id: tLuaCOM.cpp,v 1.17 2002/09/25 15:33:51 almendra Exp $";
static char *rcsname = "$Name:  $";

#include <iostream.h>

#include "tLuaCOM.h"
#include "tLuaDispatch.h"
#include "LuaAux.h"
#include "tUtil.h"
#include "tLuaCOMException.h"
#include "tCOMUtil.h"

char* tLuaCOM::TAG = "LuaCOM";

tLuaCOM::tLuaCOM(IDispatch *pdisp_arg,
                 ITypeInfo *ptinfo_arg,
                 ITypeInfo *coclass_arg,
                 LuaBeans *lbeans)
{
  connection_point = NULL;
  connection_point_cookie = NULL;

  pdisp = pdisp_arg;
  pdisp->AddRef();

  ptinfo = ptinfo_arg;
  ptinfo->AddRef();

  coclassinfo = coclass_arg;
  
  if(coclassinfo != NULL)
    coclassinfo->AddRef();

  typehandler = new tLuaCOMTypeHandler(ptinfo_arg, lbeans);
}

tLuaCOM::~tLuaCOM()
{
  if(connection_point != NULL)
    releaseConnection();

  pdisp->Release();
  pdisp = NULL;

  ptinfo->Release();
  ptinfo = NULL;

  if(coclassinfo != NULL)
  {
    COM_RELEASE(coclassinfo);
  }

  delete typehandler;
  typehandler = NULL;
}

HRESULT tLuaCOM::get_funcdescr(DISPID dispid, INVOKEKIND invkind,
                              FUNCDESC** ppfuncdesc)
{
   TYPEATTR * ptypeattr = NULL;
   HRESULT hr;
   unsigned int i, n;

   hr = ptinfo->GetTypeAttr(&ptypeattr);
   if (FAILED(hr))
   {
      return  hr;
   }

   n = ptypeattr->cFuncs;
   ptinfo->ReleaseTypeAttr(ptypeattr);

   for (i = 0; i < n;i++)
   {
      hr = ptinfo->GetFuncDesc(i, ppfuncdesc);
      if (FAILED(hr))
      {
         return  hr;
      }
      if (*ppfuncdesc == NULL)
      {
         return E_INVALIDARG;
      }

      //fazer o que tem que ser feito
      if ((*ppfuncdesc)->memid == dispid && (*ppfuncdesc)->invkind == invkind)
      {
         return S_OK;
      }

      ptinfo->ReleaseFuncDesc(*ppfuncdesc);
   }

   *ppfuncdesc = NULL;
   return E_INVALIDARG;
}

HRESULT tLuaCOM::get_dispid(const char* name, DISPID* dispid)
{
   HRESULT hr;
   wchar_t* w_name = (wchar_t*) malloc( (strlen(name) + 1) * sizeof(wchar_t));
   mbstowcs(w_name,name,strlen(name)+1);

   hr = pdisp->GetIDsOfNames(IID_NULL, &w_name, 1,
                          LOCALE_SYSTEM_DEFAULT,dispid);
   free(w_name);
   return  hr;
}

int tLuaCOM::getprop(FUNCDESC* pfuncdesc)
{
   return call(pfuncdesc, tLuaObjList());
}

int tLuaCOM::call(FUNCDESC *pFuncDesc, tLuaObjList& params)
{
   HRESULT hr             = 0;
   unsigned int i         = 0;
   UINT ArgErr            = 0;
   
   // number of return values (required to interface with Lua)
   int num_retvals        = 0;

   DISPPARAMS dispparams;
   VARIANTARG result;
   EXCEPINFO excepinfo;
      
   VariantInit(&result);
   
   // we don't support passing optional parameters through
   // safearrays

   CHECKFEATURE(pFuncDesc->cParamsOpt != -1,
     "Passing optional parameters through safe arrays not supported");


   // preenche estrutura com parametros
   typehandler->fillDispParams(dispparams, pFuncDesc, params);

   hr = pdisp->Invoke(
     pFuncDesc->memid,
     IID_NULL, 
     LOCALE_SYSTEM_DEFAULT,
     pFuncDesc->invkind,
     &dispparams,
     &result,
     &excepinfo,
     &ArgErr);

   if(SUCCEEDED(hr))
   {
     /* empilha valor de retorno */
     
     num_retvals += typehandler->com2lua(result);

     if(pFuncDesc->invkind == INVOKE_FUNC)
       num_retvals += typehandler->pushOutValues(params, pFuncDesc, dispparams);

     // Limpa parametros e valor de retorno
     typehandler->releaseVariants(&dispparams);
     VariantClear(&result);

     return num_retvals;

   }
   else // Houve erro
   {
     // Limpa parametros
     typehandler->releaseVariants(&dispparams);

     if(hr == DISP_E_EXCEPTION) // excecoes
     {
       if(excepinfo.bstrDescription != NULL)
         COM_EXCEPTION(tUtil::bstr2string(excepinfo.bstrDescription));
       else if(excepinfo.wCode != 0)
         COM_EXCEPTION(tUtil::GetErrorMessage(excepinfo.wCode));
       else if(excepinfo.scode != 0)
         COM_EXCEPTION(tUtil::GetErrorMessage(excepinfo.scode));
       else
         COM_EXCEPTION("Unknown exception");
     }
     else // Erros comuns
       COM_ERROR(tUtil::GetErrorMessage(hr));
   }
}

int tLuaCOM::putprop(FUNCDESC* pfuncdesc, stkIndex val)
{
   int result;

   tLuaObjList& params = tLuaObjList(val, 1);

   result = call(pfuncdesc, params);

   return result;
}

bool tLuaCOM::addConnection(tLuaCOM *server)
{
  HRESULT hr;
  IDispatch *pdisp_server = server->GetIDispatch();

  IConnectionPointContainer *pcpc = NULL;

  assert(connection_point == NULL);
  if(connection_point != NULL)
    return false;

  hr = pdisp->QueryInterface
    (
      IID_IConnectionPointContainer, (void **) &pcpc
    );

  if(FAILED(hr))
  {
    cout << "ERROR: Doesn't implement IConnectionPointContainer" << endl;
    return false;
  }

  {
    IID guid;

    server->GetIID(&guid);

    hr = pcpc->FindConnectionPoint(guid, &connection_point);
  }

  pcpc->Release();
  pcpc = NULL;

  if(FAILED(hr))
  {
    cout << "ERROR: Requested connection point not supported" << endl;

    return false;
  }

  hr = connection_point->Advise
    (
      (IUnknown *) pdisp_server, &connection_point_cookie
    );

  if(FAILED(hr))
  {
    cout << "ERROR: Unable to advise connection point" << endl;

    connection_point->Release();
    connection_point = NULL;

    return false;
  }

  return true;
}

void tLuaCOM::releaseConnection(void)
{
  assert(connection_point);
  if(connection_point == NULL)
    return;

  connection_point->Unadvise(connection_point_cookie);
  connection_point->Release();
  connection_point = NULL;
}

//
//  isMember
//
//    Informa se existe algum metodo ou propriedade com
//    o nome passado como parametro
//

bool tLuaCOM::isMember(const char * name)
{
  HRESULT hr;
  DISPID dumb_dispid;
  
  wchar_t* w_name = (wchar_t*) malloc( (strlen(name) + 1) * sizeof(wchar_t));

  assert(w_name);
  if(!w_name)
    return false;

  mbstowcs(w_name, name, strlen(name)+1);

  hr = pdisp->GetIDsOfNames(IID_NULL, &w_name, 1,
                        LOCALE_SYSTEM_DEFAULT, &dumb_dispid);
  free(w_name);
  w_name = NULL;

  if(!FAILED(hr))
    return true;
  else
    return false;
}


void tLuaCOM::getHelpInfo(char **ppHelpFile, unsigned long *pHelpContext)
{
  ITypeLib *typelib = NULL;
  BSTR helpfile;
  HRESULT hr = S_OK;
  
  hr = ptinfo->GetDocumentation(-1, NULL, NULL, pHelpContext, &helpfile);

  if(FAILED(hr) || helpfile == NULL)
  {
    *ppHelpFile = NULL;
    return;
  }

  // Se nao conseguiu help contextna propria interface, tenta obte-lo
  // na type library
  if(*pHelpContext == 0)
  {
    unsigned int dumb_index = 0;
    unsigned long typelib_help_context = 0;
    BSTR helpfile_typelib;

    hr = ptinfo->GetContainingTypeLib(&typelib, &dumb_index);

    if(!FAILED(hr))
    {
      hr = typelib->GetDocumentation(-1, NULL, NULL,
        &typelib_help_context, &helpfile_typelib);

      if(!FAILED(hr))
      {
        SysFreeString(helpfile);

        helpfile = helpfile_typelib;
        *pHelpContext = typelib_help_context;
      }
    }
  }

  int str_size = SysStringLen(helpfile);
  *ppHelpFile = (char *) malloc((str_size + 1)*sizeof(wchar_t));
  wcstombs(*ppHelpFile, helpfile, str_size+1);

  SysFreeString(helpfile);
}



//
// CreateLuaCOM
//


tLuaCOM * tLuaCOM::CreateLuaCOM(IDispatch * pdisp,
                                ITypeInfo* coclassinfo,
                                LuaBeans *lbeans)
{
  HRESULT hr = S_OK;

  CHECKPARAM(pdisp && lbeans);


  // tries to get some CoClass typeinfo
  if(coclassinfo == NULL)
    coclassinfo = tCOMUtil::GetCoClassTypeInfo((IUnknown*)pdisp);

  ITypeInfo* typeinfo = tCOMUtil::GetDispatchTypeInfo(pdisp);

  if(typeinfo == NULL)
  {
    if(coclassinfo != NULL)
      coclassinfo->Release();

    return NULL;
  }

  tLuaCOM *lcom = new tLuaCOM(pdisp, typeinfo, coclassinfo, lbeans);

  return lcom;
}

ITypeInfo * tLuaCOM::GetDefaultEventsInterface()
{
  if(coclassinfo == NULL)
    return NULL;
  
  return tCOMUtil::GetDefaultInterfaceTypeInfo(coclassinfo, true);
}

void tLuaCOM::ReleaseFuncDesc(FUNCDESC * pfuncdesc)
{
  ptinfo->ReleaseFuncDesc(pfuncdesc);
}

IDispatch * tLuaCOM::GetIDispatch()
{
  return pdisp;
}

void tLuaCOM::GetIID(IID * piid)
{
  TYPEATTR *ptypeattr;

  ptinfo->GetTypeAttr(&ptypeattr);
  *piid = ptypeattr->guid;
  ptinfo->ReleaseTypeAttr(ptypeattr);
}


ITypeInfo* tLuaCOM::GetCoClassTypeInfo()
{
  return coclassinfo;
}
