// tLuaCOMTypeHandler.cpp: implementation of the tLuaCOMTypeHandler class.
//
//////////////////////////////////////////////////////////////////////

// RCS Info
static char *rcsid = "$Id: tLuaCOMTypeHandler.cpp,v 1.15 2002/09/25 15:40:06 almendra Exp $";
static char *rcsname = "$Name:  $";


#include <ole2.h>

extern "C"
{
#include <lua.h>
}

#include <iostream.h>
#include <assert.h>
#include <stdio.h>

#include "tLuaCOMTypeHandler.h"
#include "tLuaCOM.h"
#include "tLuaVector.h"
#include "tLuaCOMException.h"

#include "tUtil.h"
#include "LuaAux.h"
#include "luabeans.h"

#define LUA_NOOBJECT 0


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

tLuaCOMTypeHandler::tLuaCOMTypeHandler(ITypeInfo *ptypeinfo, LuaBeans *p_lbeans)
{
  lbeans = p_lbeans;
  L = lbeans->getLuaState();
  m_typeinfo = ptypeinfo;
  m_typeinfo->AddRef();

  // gets the tag for IUnknown and caches it
  lua_getglobal(L, LUACOM_IUNKNOWN_TAGNAME);
  IUnknown_tag = (int) lua_tonumber(L, -1);
  lua_remove(L, -1);
}

tLuaCOMTypeHandler::~tLuaCOMTypeHandler()
{
  m_typeinfo->Release();
  m_typeinfo = NULL;
}

int tLuaCOMTypeHandler::com2lua(VARIANTARG varg_orig)
{
  HRESULT hr = S_OK;

    // trata SAFE ARRAY separadamente
  if(varg_orig.vt & VT_ARRAY)
  {
    safearray_com2lua(varg_orig);
    return 1;
  }

  // assumes that something will be returned
  int num_retvals = 1;

  VARIANT varg;
  VariantInit(&varg);

  // dereferences VARIANTARG (if necessary)
  hr = VariantCopyInd(&varg, &varg_orig);

  if(FAILED(hr))
    COM_ERROR(tUtil::GetErrorMessage(hr));

  // used in some type conversions
  VARIANTARG new_varg;
  VariantInit(&new_varg);

  try
  {
    switch (varg.vt)
    {
    case VT_VOID:
    case VT_EMPTY:
      // nothing will be returned
      num_retvals = 0;
      break;

    case VT_NULL:
      // SQL's NULL value.
      lua_pushnil(L);
      break;

    case VT_CY:
    case VT_UI1:
    case VT_UI2:
    case VT_UI4:
    case VT_INT:
    case VT_UINT:
    case VT_I1:
    case VT_I2:
    case VT_I4:
    case VT_R4:
    case VT_R8:
      {
        new_varg.vt = VT_R8;
        HRESULT hr = VariantChangeType(&new_varg, &varg, 0, VT_R8);
        assert(SUCCEEDED(hr));

        lua_pushnumber(L, new_varg.dblVal);

        break;
      }

    case VT_DATE:
      {
        HRESULT hr = VariantChangeType(&new_varg, &varg, 0, VT_BSTR);
        assert(SUCCEEDED(hr));

        lua_pushstring(L, (char *) tUtil::bstr2string(new_varg.bstrVal));

        break;
      }


    case VT_ERROR: 
      // assumes that a parameter has been omitted
      lua_pushnil(L);
      break;

    case VT_BOOL:
      if (varg.boolVal)
         lua_pushnumber(L, 1.);
      else
         lua_pushnumber(L, 0.);
      break;

    case VT_BSTR:
      {
        const char* str = tUtil::bstr2string(varg.bstrVal);
        lua_pushstring(L, (char *) str);
      
        break;
      }

    case VT_DISPATCH:
      {
        unsigned int ninfo = 0;
        IDispatch *pdisp = varg.pdispVal;

        if(pdisp == NULL)
        {
          lua_pushnil(L);
          break;
        }

        tLuaCOM* lcom = tLuaCOM::CreateLuaCOM(pdisp, NULL, lbeans);

        if(lcom == NULL)
        {
          lua_pushnil(L);
          break;
        }

        lbeans->push(lcom);
      }
      break;

    case VT_UNKNOWN:
      {
        pushIUnknown(varg.punkVal);
        break;
      }

    default:
      {
        static char msg[100];
      
        sprintf(msg, "COM->Lua - Type 0x%.2x not implemented.", varg.vt); 
      
        TYPECONV_ERROR(msg);
      
        break;
      }
    }
  }
  catch(class tLuaCOMException& e)
  {
    VariantClear(&varg);
    throw;
  }

  VariantClear(&varg);

  return num_retvals;
}

bool tLuaCOMTypeHandler::lua2com(TYPEDESC tdesc,
                                 stkIndex luaval,
                                 VARIANTARG * pvarg,
                                 bool variant_initialized)
{
  CHECKPARAM(luaval > 0 && pvarg != NULL);

  // removes all VT_PTR and VT_USERDEFINED indirections
  tdesc = processTYPEDESC(tdesc);

  VARTYPE vt;
  if(!variant_initialized)
    vt = tdesc.vt;
  else
    vt = pvarg->vt;

   switch (vt)
   {
   case VT_UI1:
   case VT_I2:
   case VT_I4:
   case VT_R4:
   case VT_R8:
   case VT_INT:
     {
       FromNumberToVariant(pvarg, luaval, vt);
       
       break;
     }


   case VT_CY:
     {
       // tests whether the value was passed as a string or
       // as a number and apply the corresponding type
       // conversions

       VARIANTARG varg;

       switch(lua_type(L, luaval))
       {
       case LUA_TNUMBER:
         varg.vt = VT_R8;
         varg.dblVal = lua_tonumber(L, luaval);
         break;

       case LUA_TSTRING:
         varg.vt = VT_BSTR;
         varg.bstrVal = tUtil::string2bstr(lua_tostring(L, luaval));
         break;

       default:
         {
           lua2com_conversionError(luaval, "CURRENCY");
           break;
         }
       }

       // Now, converts to CURRENCY
       HRESULT hr = VariantChangeType(pvarg, &varg, 0, VT_CY);
       VariantClear(&varg);

       if(FAILED(hr))
       {
         lua2com_conversionError(
           luaval, 
           "CURRENCY",
           "VariantChangeType failed",
           hr);
       }

       break;
     }

   case VT_DATE:
     {
       if(lua_type(L, luaval) != LUA_TSTRING)
         TYPECONV_ERROR("LuaCOM only converts strings to DATE");

       // first, converts to bstr
       VARIANTARG varg;
       varg.vt = VT_BSTR;
       varg.bstrVal = tUtil::string2bstr(lua_tostring(L, luaval));

       // then, converts to DATE
       HRESULT hr = VariantChangeType(pvarg, &varg, 0, VT_DATE);
       VariantClear(&varg);

       if(FAILED(hr))
         TYPECONV_ERROR("Cannot convert a value to DATE");

       break;
     }

  case VT_DATE | VT_BYREF:
     {
       if(lua_type(L, luaval) != LUA_TSTRING)
         TYPECONV_ERROR("LuaCOM only converts strings to DATE");

       if(!variant_initialized)
       {
         pvarg->vt = VT_DATE | VT_BYREF;
         pvarg->pdate = (DATE*) CoTaskMemAlloc(sizeof(DATE));
       }

       VARIANTARG varg;

       // first, converts to bstr
       varg.vt = VT_BSTR;
       varg.bstrVal = tUtil::string2bstr(lua_tostring(L, luaval));

       // then, converts to DATE
       VariantChangeType(&varg, &varg, 0, VT_DATE);

       *pvarg->pdate = varg.date;

       break;

     }


   case VT_UI1 | VT_BYREF:
     {
       if(!variant_initialized)
       {
         pvarg->vt = VT_UI1 | VT_BYREF;
         pvarg->pbVal = (unsigned char*) CoTaskMemAlloc(sizeof(unsigned char));
       }

       *(pvarg->pbVal) = (unsigned char)lua_tonumber(L, luaval);

       break;
     }

   case VT_I2 | VT_BYREF:
     {
       if(!variant_initialized)
       {
         pvarg->vt = VT_I2 | VT_BYREF;
         pvarg->piVal = (short*) CoTaskMemAlloc(sizeof(short));
       }

       *(pvarg->piVal) = (short)lua_tonumber(L, luaval);

       break;
     }

   case VT_I4 | VT_BYREF:
     {
       if(!variant_initialized)
       {
        pvarg->vt = VT_I4 | VT_BYREF;
        pvarg->plVal = (long*) CoTaskMemAlloc(sizeof(long));
       }

       *(pvarg->plVal) = (long)lua_tonumber(L, luaval);

       break;
     }

   case VT_R4 | VT_BYREF:
     {
       if(!variant_initialized)
       {
        pvarg->vt = VT_R4 | VT_BYREF;
        pvarg->pfltVal = (float*) CoTaskMemAlloc(sizeof(float));
       }

       *(pvarg->pfltVal) = (float)lua_tonumber(L, luaval);

       break;
     }

   case VT_R8 | VT_BYREF:
     {
       if(!variant_initialized)
       {
        pvarg->vt = VT_R8 | VT_BYREF;
        pvarg->pdblVal = (double*) CoTaskMemAlloc(sizeof(double));
       }

       *(pvarg->pdblVal) = (double)lua_tonumber(L, luaval);

       break;
     }

   case VT_CY | VT_BYREF:
     {
       if(!variant_initialized)
       {
         pvarg->vt = VT_CY | VT_BYREF;
         pvarg->pcyVal = (CY*) CoTaskMemAlloc(sizeof(CY));
       }

       VARIANTARG varg;

       FromNumberToVariant(&varg, luaval, VT_CY);

       *pvarg->pcyVal = varg.cyVal;

       break;
     }

   case VT_INT | VT_BYREF:
     {
       if(!variant_initialized)
       {
        pvarg->vt = VT_INT | VT_BYREF;
        pvarg->pintVal = (int*) CoTaskMemAlloc(sizeof(int));
       }

       *pvarg->pintVal = (int) lua_tonumber(L, luaval);

       break;
     }

   case VT_BOOL:
     {
       pvarg->vt = VT_BOOL;

       double value = lua_tonumber(L, luaval);

       if(value == 0)
         pvarg->boolVal = VARIANT_FALSE;
       else
         pvarg->boolVal = VARIANT_TRUE;

       break;
     }

   case VT_BOOL | VT_BYREF:
     {
       if(!variant_initialized)
       {
         pvarg->vt = VT_BOOL | VT_BYREF;
         pvarg->pboolVal = (VARIANT_BOOL*) CoTaskMemAlloc(sizeof(VARIANT_BOOL));
       }

       double value = lua_tonumber(L, luaval);

       if(value == 0)
         *pvarg->pboolVal = VARIANT_FALSE;
       else
         *pvarg->pboolVal = VARIANT_TRUE;

       break;
     }

   case VT_BSTR:
     {
       pvarg->vt = VT_BSTR;
       pvarg->bstrVal = tUtil::string2bstr(lua_tostring(L, luaval));

       break;
     }

   case VT_BSTR | VT_BYREF:
     {
       if(!variant_initialized)
       {
        pvarg->vt = VT_BSTR | VT_BYREF;
        pvarg->pbstrVal = (BSTR*) CoTaskMemAlloc(sizeof(BSTR));
       }

       *pvarg->pbstrVal = tUtil::string2bstr(lua_tostring(L, luaval));

       break;
     }

    case VT_VARIANT:
      {
        if (lua_type(L, luaval) == LUA_TNUMBER)
        {
           pvarg->vt = VT_R8;
           pvarg->dblVal = lua_tonumber(L, luaval);
        }
        else if (lua_type(L, luaval) == LUA_TSTRING)
        {
          const char* str = lua_tostring(L, luaval);
          long c_len = strlen(str);
          long l_len = lua_strlen(L, luaval);
          if (c_len == l_len)
          {
             pvarg->vt = VT_BSTR;
             pvarg->bstrVal = tUtil::string2bstr(str);
          }
          else
          {
             return string2safearray(str, l_len, pvarg);
          }
        }
        else if(from_lua(luaval) != NULL)
        {
          pvarg->vt = VT_DISPATCH;

          FromDispatchToVariantarg(pvarg, luaval, pvarg->vt, variant_initialized);
        }
        else if(isIUnknown(luaval))
        {
          pvarg->vt = VT_UNKNOWN;
          pvarg->punkVal = (IUnknown *) lua_touserdata(L, luaval);
          pvarg->punkVal->AddRef();
        }
        else
        {
          lua2com_conversionError(luaval, "VARIANT");
          break;
        }

        break;
      }

   case VT_VARIANT | VT_BYREF:
     {
       if(!variant_initialized)
       {
         pvarg->vt = VT_VARIANT | VT_BYREF;
         pvarg->pvarVal = (VARIANTARG*) CoTaskMemAlloc(sizeof(VARIANTARG));
       }

       TYPEDESC tdesc2;
       tdesc2.vt = VT_VARIANT;
       lua2com(tdesc2, luaval, pvarg->pvarVal, false);

       break;
     }


   case VT_DISPATCH:
   case VT_DISPATCH | VT_BYREF:
     {
       FromDispatchToVariantarg(pvarg, luaval, vt, variant_initialized);

       break;
     }

   case VT_UNKNOWN:
     {
       if(isIUnknown(luaval))
       {
         pvarg->vt = VT_UNKNOWN;
         pvarg->punkVal = (IUnknown *) lua_touserdata(L, luaval);
         pvarg->punkVal->AddRef();
       }
       else
         TYPECONV_ERROR("Attemp to convert a non-IUnknown userdata to IUnknown");

       break;
     }

   case VT_UNKNOWN | VT_BYREF:
     {
       if(isIUnknown(luaval))
       {
         if(!variant_initialized)
         {
           pvarg->vt = VT_UNKNOWN | VT_BYREF;
           pvarg->ppunkVal = (IUnknown**) CoTaskMemAlloc(sizeof(IUnknown*));
         }

         *pvarg->ppunkVal = (IUnknown *) lua_touserdata(L, luaval);
         (*pvarg->ppunkVal)->AddRef();
       }
       else
         TYPECONV_ERROR("Attemp to convert a non-IUnknown userdata to IUnknown");

       break;
     }

   case VT_SAFEARRAY:
     {
       safearray_lua2com(*tdesc.lptdesc, luaval, pvarg);
       break;
     }

   default:
     {
       static char msg[100];

       sprintf(msg, "Lua->COM - Type 0x%.2x not implemented.", vt); 

       TYPECONV_ERROR(msg);

       break;
     }
   }

   return true;
}

bool tLuaCOMTypeHandler::setRetval(const FUNCDESC * funcdesc,
                                   stkIndex luaval,
                                   VARIANTARG * pvarg)
{
  if(funcdesc->elemdescFunc.tdesc.vt != VT_VOID &&
     funcdesc->elemdescFunc.tdesc.vt != VT_HRESULT 
      )
  {
    lua2com(funcdesc->elemdescFunc.tdesc, luaval, pvarg, false);
  }
  else
    return false;

  return true;
}


int tLuaCOMTypeHandler::pushOutValues(tLuaObjList& params,
                                      const FUNCDESC * funcdesc,
                                      const DISPPARAMS& dispparams)
{
  const int num_args = funcdesc->cParams;
  int i = 0;
  int num_pushed_values = 0;
   
  // Procura valor de retorno dos parametros de saida
  for(i = 0; i < num_args; i++)
  {
    // aliases para simplificar digitacao
    const USHORT& paramflags =
      funcdesc->lprgelemdescParam[i].paramdesc.wParamFlags;

    const TYPEDESC tdesc = 
      processTYPEDESC(funcdesc->lprgelemdescParam[i].tdesc);

    if( ( paramflags == PARAMFLAG_NONE && (tdesc.vt & VT_BYREF) ) || 
        ( paramflags & PARAMFLAG_FOUT )
      )
    {
      num_pushed_values+= com2lua(dispparams.rgvarg[num_args - i - 1]);
    }
  }

  return num_pushed_values;
}


void tLuaCOMTypeHandler::releaseVariants(DISPPARAMS *pDispParams)
{
  unsigned int i = 0;
  VARIANTARG* &vargs = pDispParams->rgvarg;

  if (vargs != NULL)
  {
    for (i = 0; i < pDispParams->cArgs; i ++)
    {
      releaseVariant(&vargs[i]);
    }

    delete [] vargs;

    vargs = NULL;
    pDispParams->cArgs = 0;
  }

}

HRESULT tLuaCOMTypeHandler::FromNumberToVariant(VARIANTARG *pvarg, stkIndex luaval, VARTYPE vt)
{
  HRESULT hr = S_OK;
       
  pvarg->dblVal = lua_tonumber(L, luaval);
  pvarg->vt = VT_R8;

  hr = VariantChangeType(pvarg, pvarg, 0, vt);
  
  if(FAILED(hr))
    TYPECONV_ERROR("Error converting number");

  return hr;
}

void tLuaCOMTypeHandler::FromDispatchToVariantarg(VARIANTARG *pvarg,
                                                  stkIndex luaval,
                                                  VARTYPE vt,
                                                  bool initialized)
{
  CHECKPRECOND(vt == VT_DISPATCH || vt == (VT_DISPATCH | VT_BYREF));

  tLuaCOM *lcom = from_lua(luaval);

  if(!lcom)
  {
    TYPECONV_ERROR("Not an Interface");
  }

  IDispatch *pdisp = lcom->GetIDispatch();
  
  pdisp->AddRef();

  pvarg->vt = vt;

  if(vt & VT_BYREF)
  {
    if(!initialized)
      pvarg->ppdispVal = (IDispatch**) CoTaskMemAlloc(sizeof(IDispatch*));

    *pvarg->ppdispVal = pdisp;
  }
  else
  {
    pvarg->pdispVal = pdisp;
  }
}

//
// Preenche estrutura DISPPARAMS, inicializando parametros
//

void tLuaCOMTypeHandler::fillDispParams(DISPPARAMS& rDispParams,
                                        FUNCDESC * pFuncDesc,
                                        tLuaObjList& params)
{
  // initializes structure
  rDispParams.cArgs = 0;
  rDispParams.cNamedArgs = 0;
  rDispParams.rgvarg = NULL;
  rDispParams.rgdispidNamedArgs = NULL;

  // se funcao nao recebe parametros...
  if (pFuncDesc->cParams == 0)
    return;


  static DISPID dispidNamed = DISPID_PROPERTYPUT;

  unsigned short i          = 0;
  stkIndex val              = -1;
  unsigned short max_params = pFuncDesc->cParams;

  // referencias para simplificar nomes
  unsigned int& r_cArgs   = rDispParams.cArgs; 
  VARIANTARG* &r_rgvarg   = rDispParams.rgvarg;

  // caso particular do propertyput
  if(pFuncDesc->invkind == DISPATCH_PROPERTYPUT ||
     pFuncDesc->invkind == DISPATCH_PROPERTYPUTREF)
  {
    rDispParams.cNamedArgs = 1;
    rDispParams.rgdispidNamedArgs = &dispidNamed;
  }

  r_cArgs = 0; // comeca vazio
  long lua_args = 0;


  // cria array com tamanho suficiente para numero maximo
  // de parametros

  r_rgvarg = new VARIANTARG[max_params]; 


  // itera no array lprgelemdescParam procurando pegar
  // os parametros da tabela lua

  try
  {
    for (i = 0; i < max_params; i++)
    {
      VariantInit(&r_rgvarg[r_cArgs]);
      val = params.getparam(lua_args);

      PARAMDESC paramdesc = pFuncDesc->lprgelemdescParam[i].paramdesc;
      bool hasdefault = false;

      if(paramdesc.wParamFlags & PARAMFLAG_FHASDEFAULT)
      {
        hasdefault = true;
        paramdesc.wParamFlags &= ~PARAMFLAG_FHASDEFAULT;
      }

      switch(paramdesc.wParamFlags)
      {
        // cases where a in-param is expected
      case PARAMFLAG_NONE:

      case PARAMFLAG_FIN:
      case PARAMFLAG_FIN | PARAMFLAG_FOUT:

      case PARAMFLAG_FIN | PARAMFLAG_FOPT:
      case PARAMFLAG_FIN | PARAMFLAG_FOUT | PARAMFLAG_FOPT:
        {
          if(val != 0 && lua_type(L, val) != LUA_TNONE && !lua_isnil(L, val)) 
          {
            lua2com(
              pFuncDesc->lprgelemdescParam[i].tdesc,
              val,
              &r_rgvarg[r_cArgs],
              false
              );
          }
          else if(hasdefault)
          {
            VariantCopy(&r_rgvarg[r_cArgs], &paramdesc.pparamdescex->varDefaultValue);
          }
          else
          {
            // assumes that a parameter is expected but has not been found

            r_rgvarg[r_cArgs].vt = VT_ERROR;
            r_rgvarg[r_cArgs].scode = DISP_E_PARAMNOTFOUND;
          }

          r_cArgs++;
          lua_args++;

          break;
        }

        // does nothing for out parameters
      case PARAMFLAG_FOUT:
      case PARAMFLAG_FOUT | PARAMFLAG_FOPT:
        {
          r_cArgs++;

          break;
        }
    
        // outros casos nao sao suportados
      default:
        TYPECONV_ERROR("Unsupported parameter flags.");
        break;
      }
    }
  }
  catch(class tLuaCOMException& e)
  {
    UNUSED(e);

    delete r_rgvarg;
    r_rgvarg = NULL;
    throw;
  }

  // inverte ordem dos parametros
  if(r_cArgs > 0)
  {
    VARIANTARG temp;

    for(i = 0; i < r_cArgs/2; i++)
    {
      temp = r_rgvarg[i];
      r_rgvarg[i] = r_rgvarg[r_cArgs - i - 1]; 
      r_rgvarg[r_cArgs - i - 1] = temp;
    }
  }

  return;
}


void tLuaCOMTypeHandler::pushLuaArgs(const DISPPARAMS* pDispParams,
                                     const ELEMDESC* pElemDesc)
{
  unsigned int arg = 0;

  for(arg = 0; arg < pDispParams->cArgs; arg++)
  {
    const USHORT& wParamFlags = pElemDesc[arg].paramdesc.wParamFlags;
    const TYPEDESC& tdesc = pElemDesc[arg].tdesc;

    if(wParamFlags & PARAMFLAG_FIN || (wParamFlags == PARAMFLAG_NONE))
    {
      com2lua(pDispParams->rgvarg[pDispParams->cArgs - arg - 1]);
    }
  }
}

void tLuaCOMTypeHandler::setOutValues(FUNCDESC * pFuncDesc,
                                      DISPPARAMS * pDispParams,
                                      stkIndex outvalue
                                      )
{
  const int num_args = pFuncDesc->cParams;
  int i = 0;

  // Procura valor de retorno dos parametros de saida
  for(i = 0; i < num_args; i++)
  {
    // aliases para simplificar digitacao

    const TYPEDESC tdesc = 
      processTYPEDESC(pFuncDesc->lprgelemdescParam[i].tdesc);

    const USHORT& paramflags =
      pFuncDesc->lprgelemdescParam[i].paramdesc.wParamFlags;

    if((paramflags == PARAMFLAG_NONE && (tdesc.vt & VT_BYREF))
      || ( paramflags & PARAMFLAG_FOUT )
      )
    {
      if(outvalue == 0 || lua_type(L, outvalue) == LUA_TNONE)
        break; // acabou lista de valores de retorno

      // clears variant first, releasing any memory allocated
      releaseVariant(&pDispParams->rgvarg[num_args - i - 1]);

      lua2com(
        tdesc,
        outvalue,
        &pDispParams->rgvarg[num_args - i - 1],
        false
        );

      outvalue++;
    }
  }

}

//
// Conversao de Safe Arrays para tabelas lua e vice versa
//

//  funcoes auxiliares

// funcoes auxiliares de safearray_lua2com
namespace safearray_aux
{
  long * dimensionsFromBounds(SAFEARRAYBOUND* bounds, long num_bounds);
  void put_in_array(
    SAFEARRAY* safearray,
    VARIANT var_value,
    long* indices,
    VARTYPE vt);

  void inc_indices(long *indices, SAFEARRAYBOUND *bounds, unsigned long dimensions);
  SAFEARRAYBOUND* getRightOrderedBounds(
    SAFEARRAYBOUND *bounds, 
    unsigned long num_dimensions);
};


SAFEARRAYBOUND* safearray_aux::getRightOrderedBounds(
    SAFEARRAYBOUND *bounds, 
    unsigned long num_dimensions)
{
  SAFEARRAYBOUND* new_bounds = new SAFEARRAYBOUND[num_dimensions];

  unsigned long i = 0;

  for(i = 0; i < num_dimensions; i++)
    new_bounds[i] = bounds[num_dimensions - i - 1];

  return new_bounds;
}


void safearray_aux::put_in_array(SAFEARRAY* safearray,
                         VARIANT var_value,
                         long* indices,
                         VARTYPE safearray_type
                         )
{
  HRESULT hr = S_OK;

  if(safearray_type == VT_VARIANT)
  {
    hr = SafeArrayPutElement(safearray, indices, &var_value);
  }
  else
  {
    switch(var_value.vt)
    {
     case VT_UI1:
     case VT_I2:
     case VT_I4:
     case VT_R4:
     case VT_R8:
     case VT_CY:
     case VT_DATE:
     case VT_INT:
     case VT_BOOL:
      hr = SafeArrayPutElement(safearray, indices, &var_value.dblVal);
      break;

    case VT_BSTR:
    case VT_DISPATCH:
      hr = SafeArrayPutElement(safearray, indices, var_value.bstrVal);
      break;

    default:
      LUACOM_EXCEPTION(INTERNAL_ERROR);
      break;
    }
  }

  if(FAILED(hr))
    LUACOM_EXCEPTION(INTERNAL_ERROR);
}

stkIndex tLuaCOMTypeHandler::get_from_array(SAFEARRAY* safearray,
                                         long *indices,
                                         const VARTYPE& vt
                                         )
{
  VARIANTARG varg;
  void *pv = NULL;

 
  HRESULT hr = S_OK;

  if(vt == VT_VARIANT)
  {
    pv = &varg;
  }
  else
  {
    VariantInit(&varg);
    varg.vt = vt;

    // e' uma union, tanto faz de quem pego o ponteiro
    pv = (void *) &varg.dblVal; 
  }

  hr = SafeArrayGetElement(safearray, indices, pv);

  if(FAILED(hr))
    LUACOM_EXCEPTION(INTERNAL_ERROR);

  int converted = com2lua(varg);

  VariantClear(&varg);

  if(converted == 0)
    return LUA_NOOBJECT;

  return lua_gettop(L);
}



void safearray_aux::inc_indices(long *indices, 
                        SAFEARRAYBOUND *bounds,
                        unsigned long dimensions
                        )
{
  unsigned long j = 0;

  indices[0]++;
  j = 0;

  while(
    (indices[j] >= (long) bounds[j].cElements) &&
    (j < (dimensions - 1))
    )
  {
    indices[j] = 0;
    indices[j+1]++;

    j++;
  }
}


//
// Cuida da conversao de tabelas para safe arrays
//

void tLuaCOMTypeHandler::safearray_lua2com(const TYPEDESC & tdesc,
                                           stkIndex luaval,
                                           VARIANTARG * pvarg)
{
  using namespace safearray_aux;

  CHECKPARAM(pvarg);
  
  // cria LuaVector baseado na tabela passada
  tLuaVector luavector(lbeans);

  luavector.InitVectorFromTable(luaval);

  // Cria variaveis
  unsigned long i = 0;
  const unsigned long dimensions = luavector.get_Dimensions();
  SAFEARRAYBOUND *bounds = new SAFEARRAYBOUND[dimensions];
  SAFEARRAY *safearray = NULL;
  VARIANTARG var_value;

  VariantInit(&var_value);


  // inicializa dimensoes
  for(i = 0; i < dimensions; i++)
  {
    bounds[i].lLbound = 0;
    bounds[i].cElements = luavector.get_Nth_Dimension(dimensions - i);
  }


  // cria array
  safearray = SafeArrayCreate(tdesc.vt, dimensions, bounds);
  
  long *indices = NULL;
  
  try
  {
    CHECK(safearray, INTERNAL_ERROR);

    // Inicializa indices
    indices = new long[dimensions];

    for(i = 0; i < dimensions; i++)
      indices[i] = 0;

    // copia elementos um por um
    while(indices[dimensions - 1] < (long) bounds[dimensions - 1].cElements)
    {
      // obtem valor
      luaval = luavector.getindex(indices, dimensions);

      //converte
      lua2com(tdesc, luaval, &var_value, false);

      // coloca no array
      put_in_array(safearray, var_value, indices, tdesc.vt);

      // libera
      VariantClear(&var_value);

      // incrementa indices
      inc_indices(indices, bounds, dimensions);
    }
  }
  catch(class tLuaCOMException&)
  {
    delete bounds;
    delete indices;
    SafeArrayDestroy(safearray);

    throw;
  }


  // preenche variantarg
  pvarg->vt = tdesc.vt | VT_ARRAY;
  pvarg->parray = safearray;


  // libera memoria
  delete bounds;
  delete indices;

  return;
}

bool tLuaCOMTypeHandler::string2safearray(const char* str, long len, VARIANTARG * pvarg)
{
  using namespace safearray_aux;

  CHECK(pvarg, PARAMETER_OUT_OF_RANGE);
  HRESULT hr = S_OK;
  
  // cria array
  SAFEARRAY *safearray = SafeArrayCreateVector(VT_UI1, 0, len);
  CHECK(safearray, INTERNAL_ERROR);

  void * buffer = NULL;
  hr = SafeArrayAccessData(safearray,&buffer);
  if(FAILED(hr))
    LUACOM_EXCEPTION(INTERNAL_ERROR);
  if (buffer != NULL)
     memcpy(buffer,str,len);
  SafeArrayUnaccessData(safearray);

  // preenche variantarg
  pvarg->vt = VT_UI1 | VT_ARRAY;
  pvarg->parray = safearray;

  return true;
}

long * safearray_aux::dimensionsFromBounds(SAFEARRAYBOUND* bounds,
                                           long num_bounds
                                           )
{
  int i = 0;
  long *dimensions = new long[num_bounds];

  for(i = 0; i < num_bounds; i++)
  {
    dimensions[i] =
      bounds[num_bounds - i - 1].lLbound + bounds[num_bounds - i - 1].cElements; 
  }

  return dimensions;
}



void tLuaCOMTypeHandler::safearray_com2lua(VARIANTARG & varg)
{
  using namespace safearray_aux;

  CHECK(varg.vt & VT_ARRAY, PARAMETER_OUT_OF_RANGE);

  bool succeeded          = false;
  long *indices           = NULL;
  SAFEARRAYBOUND* bounds  = NULL;
  
  try
  {
    SAFEARRAY* safearray = varg.parray;

    // pega dimensoes
    const int num_dimensions = SafeArrayGetDim(safearray);

    bounds = getRightOrderedBounds
      (
      safearray->rgsabound,
      num_dimensions
      );
  
    
      // cria objeto LuaVector
    tLuaVector luavector(lbeans);

    {
      long *dimensions = dimensionsFromBounds(bounds, num_dimensions);

      try
      {
        luavector.InitVectorFromDimensions(dimensions, num_dimensions);
      }
      catch(class tLuaCOMException&)
      {
        delete dimensions;
        throw;
      }

      delete dimensions;
    }

    // Inicializa indices
    indices = new long[num_dimensions];

    int i = 0;
    for(i = 0; i < num_dimensions; i++)
      indices[i] = 0;

    // extrai tipo de dado do array
    VARTYPE vt = varg.vt & ~VT_ARRAY;

    // holds index to lua objects
    stkIndex luaval = 0;

    // saves current stack position
    stkIndex stacktop = lua_gettop(L);

    // copia elementos um por um
    while(indices[num_dimensions-1] < (long) bounds[num_dimensions-1].cElements)
    {
      // pega do array
      luaval = get_from_array(safearray, indices, vt);

      // seta no luavector
      luavector.setindex(luaval, indices, num_dimensions);

      // incrementa indices
      inc_indices(indices, bounds, num_dimensions);
    }

    // tries to create lua table on the top of stack
    succeeded = luavector.CreateTable();

    // remove temporary objects
    stkIndex clean_until = lua_gettop(L);

    if(succeeded)
      clean_until--; // doesn't clean created table!

    while(clean_until > stacktop)
    {
      lua_remove(L, clean_until);
      clean_until--;
    }
  }
  catch(class tLuaCOMException&)
  {
    delete bounds;
    delete indices;
    throw;
  }

  delete bounds;
  delete indices;

  return;
}


tLuaCOM * tLuaCOMTypeHandler::from_lua(int index)
{
  return (tLuaCOM *) lbeans->from_lua(index);
}

TYPEDESC tLuaCOMTypeHandler::processPTR(const TYPEDESC &tdesc)
{
  CHECKPRECOND(tdesc.vt == VT_PTR);

  TYPEDESC pointed_at;

  // continues indirection
  pointed_at.vt = tdesc.lptdesc->vt;
  pointed_at.lptdesc = tdesc.lptdesc->lptdesc;

  // removes aliases
  pointed_at = processAliases(pointed_at);
  
  // if the referenced type is userdefined, gets its
  // definition
  bool userdef = false;

  if(pointed_at.vt == VT_USERDEFINED)
  {
    userdef = true;
    pointed_at = processUSERDEFINED(pointed_at);
  }

  if(userdef == true &&
     (pointed_at.vt == VT_DISPATCH || pointed_at.vt == VT_UNKNOWN))
  {
    // does nothing, because it's a VT_USERDEFINED TYPEDESC that
    // describes an interface that inherits from IDispatch.
    // Pointers (that is, single indirection) to IDispatch 
    // are always VT_DISPATCH.
  }
  else if(pointed_at.vt == VT_PTR)
  {
    // continues indirection
    pointed_at = processPTR(pointed_at);

    // We arrive here if the TYPEDESC describes a
    // pointer to a pointer. This only happens
    // when we are refencing interfaces. Since
    // interfaces are always refenced as pointers,
    // it looks like a single indirection

    pointed_at.vt |= VT_BYREF; 
  }
  else // other types under a VT_PTR are just BYREF
  {
    pointed_at.vt |= VT_BYREF; 
  }

  return pointed_at;
}

TYPEDESC tLuaCOMTypeHandler::processUSERDEFINED(const TYPEDESC &tdesc)
{
  HRESULT hr = S_OK;
  ITypeInfo *userdef = NULL;
  TYPEATTR *typeattr = NULL;
  TYPEDESC newtdesc;

  newtdesc.vt = 0;

  hr = m_typeinfo->GetRefTypeInfo(tdesc.hreftype, &userdef);

  if(FAILED(hr))
    TYPECONV_ERROR("Could not understand user-defined type");

  hr = userdef->GetTypeAttr(&typeattr);

  if(FAILED(hr))
  {
    userdef->Release();
    TYPECONV_ERROR("Could not understand user-defined type");
  }
  
  switch(typeattr->typekind)
  {
  case TKIND_ENUM:
    newtdesc.vt = VT_INT;
    break;

  case TKIND_DISPATCH:
    newtdesc.vt = VT_DISPATCH;
    break;

  case TKIND_ALIAS:
    // shouldn't arrive here: aliases must be removed via
    // processAliases()
    INTERNAL_ERROR();
    break;

  case TKIND_INTERFACE:
    newtdesc.vt = VT_UNKNOWN;
    break;

  case TKIND_UNION:
    TYPECONV_ERROR("Union type not supported!");
    break;

  case TKIND_COCLASS:
    TYPECONV_ERROR("CoClass type not supported!");
    break;

  case TKIND_RECORD:
    TYPECONV_ERROR("Record type not supported!");
    break;

  case TKIND_MODULE:
  case TKIND_MAX:
    TYPECONV_ERROR("TKIND_MODULE and TKIND_MAX not supported!");
    break;

  default:
    TYPECONV_ERROR("Unknown TYPEKIND on VT_USERDEFINED TYPEDESC");
    break;
  }

  userdef->ReleaseTypeAttr(typeattr);
  userdef->Release();

  return newtdesc;
}

//
// Clears a VARIANT, releasing first the memory allocated

void tLuaCOMTypeHandler::releaseVariant(VARIANTARG *pvarg)
{
  if(pvarg->vt & VT_BYREF && pvarg->byref != NULL)
  {
    switch(pvarg->vt & (~VT_BYREF))
    {
    case VT_UI1:
      CoTaskMemFree(pvarg->pbVal);
      break;

    case VT_I2:
      CoTaskMemFree(pvarg->piVal);
      break;

    case VT_I4:
      CoTaskMemFree(pvarg->plVal);
      break;

    case VT_R4:
      CoTaskMemFree(pvarg->pfltVal);
      break;

    case VT_R8:
      CoTaskMemFree(pvarg->pdblVal);
      break;

    case VT_BOOL:
      CoTaskMemFree(pvarg->pboolVal);
      break;

    case VT_BSTR:
      SysFreeString(*pvarg->pbstrVal);
      CoTaskMemFree(pvarg->pbstrVal);
      break;

    case VT_DISPATCH:
      (*pvarg->ppdispVal)->Release();
      CoTaskMemFree(pvarg->ppdispVal);
      break;

    case VT_UNKNOWN:
      (*pvarg->ppunkVal)->Release();
      CoTaskMemFree(pvarg->ppunkVal);
      break;

    case VT_DATE:
      CoTaskMemFree(pvarg->pdate);
      break;

    case VT_CY:
      CoTaskMemFree(pvarg->pcyVal);
      break;

    case VT_VARIANT:
      // a variant cannot contain another BYREF
      // so we just clear with VariantClear
      VariantClear(pvarg->pvarVal);
      CoTaskMemFree(pvarg->pvarVal);
      break;

    case VT_INT:
      CoTaskMemFree(pvarg->pintVal);
      break;

    default: // does nothing
      assert(0); // shouldn't arrive here...
      break;
    }

    pvarg->vt = VT_EMPTY;
    pvarg->byref = NULL;
  }
  else
    VariantClear(pvarg);

}


// Dereferences typedef's in type descriptions

TYPEDESC tLuaCOMTypeHandler::processAliases(const TYPEDESC &tdesc)
{
  // if it's not a userdefined type, does nothing
  if(tdesc.vt != VT_USERDEFINED)
    return tdesc;

  HRESULT hr = S_OK;
  ITypeInfo *userdef = NULL;
  TYPEATTR *typeattr = NULL;
  TYPEDESC newtdesc;

  newtdesc.vt = 0;

  hr = m_typeinfo->GetRefTypeInfo(tdesc.hreftype, &userdef);

  if(FAILED(hr))
    TYPECONV_ERROR("Could not understand user-defined type");

  hr = userdef->GetTypeAttr(&typeattr);

  if(FAILED(hr))
  {
    userdef->Release();
    TYPECONV_ERROR("Could not understand user-defined type");
  }

  if(typeattr->typekind == TKIND_ALIAS)
  {
    newtdesc = typeattr->tdescAlias;
    newtdesc = processAliases(newtdesc);
  }
  else
    newtdesc = tdesc;

  userdef->ReleaseTypeAttr(typeattr);
  userdef->Release();

  return newtdesc;
}

TYPEDESC tLuaCOMTypeHandler::processTYPEDESC(TYPEDESC tdesc)
{
  // removes aliases
  tdesc = processAliases(tdesc);

  bool done = false;

  switch(tdesc.vt)
  {
  case VT_USERDEFINED:
    tdesc = processUSERDEFINED(tdesc);
    break;

  case VT_PTR:
    tdesc = processPTR(tdesc);
    break;
  }

  CHECKPOSCOND(tdesc.vt != VT_USERDEFINED && tdesc.vt != VT_PTR);

  return tdesc;
}


/*
 * IsIUnknown
 *
 *   checks whether the lua value is of tag LuaCOM_IUnknown
 */

bool tLuaCOMTypeHandler::isIUnknown(stkIndex value)
{
  int value_tag = lua_tag(L, value);

  return IUnknown_tag == value_tag;
}


void tLuaCOMTypeHandler::pushIUnknown(IUnknown *punk)
{
  lua_pushusertag(L, (void *) punk, IUnknown_tag);
}

void tLuaCOMTypeHandler::lua2com_conversionError(stkIndex luaval,
                                                 const char* com_type,
                                                 const char* reason,
                                                 const HRESULT hr)
{
  CHECKPRECOND(com_type);

  static char msg[1000];

  if(reason == NULL && hr == S_OK)
  {
    sprintf(msg, "Cannot convert a %s (tag = %d) to %s",
      lua_typename(L, lua_type(L, luaval)), lua_tag(L, luaval), com_type);
  }
  else if(reason != NULL)
  {
    if(hr == S_OK)
    {
      sprintf(msg, "Cannot convert a %s (tag = %d) to %s - %s.",
        lua_typename(L, lua_type(L, luaval)), lua_tag(L, luaval), com_type, reason);
    }
    else
    {
      sprintf(msg, "Cannot convert a %s (tag = %d) to %s - %s | 0x%.8x %s",
        lua_typename(L, lua_type(L, luaval)), lua_tag(L, luaval),
        com_type, reason, hr, tUtil::GetErrorMessage(hr));

    }
  }
           
  TYPECONV_ERROR(msg);
}
