/*
 * LUACOM.cpp
 *
 *   Bind de Lua com COM/OLE
 *
 *  Renato Cerqueira
 *  Vinicius Almendra
 *
 */

// RCS Info
static char *rcsid = "$Id: luacom.cpp,v 1.20 2001-05-18 08:59:20-03 almendra Exp almendra $";
static char *rcsname = "$Name:  $";
static char *g_version = "1.0 (BETA)";

#include <string.h>
#include <iostream.h>

#include <ole2.h>
#include <ocidl.h>

#include <assert.h>
#include <stdio.h>
#include <math.h>

extern "C"
{
#include <lua.h>
#include <lauxlib.h>
#include <luadebug.h>
}

#include "luabeans.h"
#include "luacom.h"
#include "tLuaDispatch.h"
#include "tLuaCOM.h"
#include "tLuaCOMException.h"

#include "LuaAux.h"
#include "tUtil.h"


////////////////////////
//                    //
//  FUNCOES INTERNAS  //
//                    //
////////////////////////

/*
 * Carrega uma Typelib associada a um ProgID
 */

static ITypeLib *loadTypeLibFromProgID(char *pcProgID, int version_major)
{
  wchar_t* wcClsid;
  wchar_t* wcProgId;
  CLSID clsid;
  HRESULT hr;
  ITypeLib *ptlib = NULL;

  /* converte ProgID para OLESTR */
  wcProgId = (wchar_t*) malloc( (strlen(pcProgID) + 1) * sizeof(wchar_t));
  mbstowcs(wcProgId, pcProgID, strlen(pcProgID)+1);

  hr = CLSIDFromProgID(wcProgId, &clsid);

  free(wcProgId);
  wcProgId = NULL;

  if (FAILED(hr))
    return NULL;

  hr = StringFromCLSID(clsid, &wcClsid);
  if (FAILED(hr))
    return NULL;

  /* converte CLSID para string normal */
  char* pcClsid = (char*) malloc( (wcslen(wcClsid) + 1) * sizeof(char));
  wcstombs(pcClsid, wcClsid,wcslen(wcClsid)+1);

  DWORD size = 38*3; /*{F37C8063-4AD5-101B-B826-00DD01103DE1}*/
  BYTE *bLibID = (BYTE *) malloc(size);
  HKEY iid_key, obj_key, typelib_key;

  /* extrai do registry type library (GUID e versao) */

  RegOpenKeyEx(HKEY_CLASSES_ROOT,"CLSID", 0, KEY_READ, &iid_key);
  RegOpenKeyEx(iid_key, pcClsid, 0, KEY_READ, &obj_key);

  free(pcClsid);
  pcClsid = NULL;

  RegOpenKeyEx(obj_key, "TypeLib",0, KEY_READ, &typelib_key);
  RegQueryValueEx(typelib_key, NULL, NULL, NULL, bLibID, &size);

  RegCloseKey(iid_key);
  RegCloseKey(obj_key);
  RegCloseKey(typelib_key);

  wchar_t* wcTypelib= (wchar_t*) 
    malloc( (strlen((char *) bLibID) + 1) * sizeof(wchar_t));
  mbstowcs(wcTypelib, (char *) bLibID, strlen((char *) bLibID)+1);
  free(bLibID);

  CLSID libid;
  CLSIDFromString(wcTypelib, &libid);
  free(wcTypelib);

  hr = LoadRegTypeLib(libid, version_major, 0, 0, &ptlib);

  if(FAILED(hr))
    return NULL;

  return ptlib;
}

/*
 *  loadTypeLibFromName
 *  Carrega uma typelib a partir de um arquivo TLB
 */

static ITypeLib *loadTypeLibFromName(char *pcFilename)
{
  HRESULT hr;
  ITypeLib *ptlib;
  wchar_t *wcFilename = NULL;
  
  wcFilename = (wchar_t *) malloc((strlen(pcFilename)+1)*sizeof(wchar_t));
  mbstowcs(wcFilename, pcFilename, strlen(pcFilename)+1);

  hr = LoadTypeLibEx(wcFilename, REGKIND_NONE, &ptlib);

  if(FAILED(hr))
    return NULL;

  return ptlib;
}


///////////////////////////////////
//                               //
//  FUNCOES EXPORTADAS PARA LUA  //
//                               //
///////////////////////////////////

/*
 * lc_ShowHelp
 *
 * Parametros: 
 *
 *  (1) objeto luacom
 *
 *   Mostra Help da type library associada ao objeto
 *   luacom
 */

static void lc_ShowHelp(void)
{
  char *pHelpFile = NULL; 
  unsigned long context = 0;

  tLuaCOM* luacom = (tLuaCOM*) 
    tLuaCOM::lbeans->from_lua(lua_getparam(1));

  if(luacom == NULL)
    return;

  luacom->getHelpInfo(&pHelpFile, &context);

  if(pHelpFile != NULL)
  {
    if(context != 0)
      WinHelp(NULL, pHelpFile, HELP_CONTEXT, context);
    else
      WinHelp(NULL, pHelpFile, HELP_FINDER, 0);
  }
}

/*
 * lc_Connect
 *
 *   Parametros (Lua):
 *
 *    objeto luacom
 *    tabela que implementara a interface de conexao
 *
 *   Cria um connection point utilizando a tabela
 *   passada como parametro e o conecta com o objeto
 *   passado tambem como parametro
 */

static void lc_Connect(void)
{
  tLuaDispatch *client_disp = NULL;
  IID iid = CLSID_NULL;
  ITypeInfo *pTypeinfo = NULL;
  int ref = 0;

  tLuaCOM* server = (tLuaCOM*) 
    tLuaCOM::lbeans->from_lua(lua_getparam(1));

  if(!server)
  {
    lua_pushnil();
    return;
  }

  /* obtem referencia e trava tabela lua */
  lua_pushobject(lua_getparam(2));
  ref = lua_ref(1);

  pTypeinfo = server->GetDefaultEventsInterface();

  if(pTypeinfo == NULL)
  {
    lua_pushnil();
    return;
  }

  client_disp = new tLuaDispatch(pTypeinfo, ref);

  tLuaCOM* client = tLuaCOM::CreateLuaCOM(client_disp);

  if(client == NULL)
  {
    delete client_disp;
    lua_pushnil();
    return;
  }

  server->addConnection(client);

  tLuaCOM::lbeans->push(client); 
}

/*
 *  lc_ImplInterfaceFromTypelib (Lua2C)
 *  Implementa uma interface descrida em uma type library
 *  dada
 */

static void lc_ImplInterfaceFromTypelib(void)
{
  tLuaDispatch *iluacom = NULL;
  char* typelib_name = lua_getstring(lua_getparam(2));
  char* pcInterface = lua_getstring(lua_getparam(3));
  ITypeLib *ptlib;
  
  /* obtem referencia e trava tabela lua */
  lua_pushobject(lua_getparam(1));
  int ref = lua_ref(1);

  ptlib = loadTypeLibFromName(typelib_name);

  if(ptlib == NULL)
  {
    lua_pushnil();
    return;
  }

  iluacom = tLuaDispatch::CreateLuaDispatch(ptlib, pcInterface, ref);

  ptlib->Release();
  ptlib = NULL;

  if(iluacom == NULL)
  {
    lua_pushnil();
    return;
  }

  tLuaCOM* lcom = tLuaCOM::CreateLuaCOM(iluacom);

  if(!lcom)
  {
    delete iluacom;
    lua_pushnil();
    return;
  }

  tLuaCOM::lbeans->push(lcom); 
}

/* 
 *  lc_ImplInterface (Lua2C)
 *
 *  Cria uma implementacao IDispatch para um
 *  objeto Lua dado um ProgID e o nome da
 *  interface
 *
 *  In: Implementation: table,
 *      ProgID: string,
 *      Interface: string,
 *      version_major: number
 *
 *  Out: LuaCOM_obj (table with tag LuaCOM)
*/

static void lc_ImplInterface(void)
{
  tLuaDispatch *iluacom = NULL;
  char* pcProgID = lua_getstring(lua_getparam(2));
  char* pcInterface = lua_getstring(lua_getparam(3));
  int iVersion_major = (int) lua_getnumber(lua_getparam(4));
  ITypeLib *ptlib = NULL;

  /* obtem referencia e trava tabela lua */
  lua_pushobject(lua_getparam(1));
  int ref = lua_ref(1);

  /* obtem typelib */
  ptlib = loadTypeLibFromProgID(pcProgID, iVersion_major);

  if(ptlib == NULL)
  {
    lua_pushnil();
    return;
  }

  iluacom = tLuaDispatch::CreateLuaDispatch(ptlib, pcInterface, ref);
  
  ptlib->Release();
  ptlib = NULL;

  if(iluacom == NULL)
  {
    lua_pushnil();
    return;
  }

  tLuaCOM* lcom = tLuaCOM::CreateLuaCOM(iluacom);

  if(lcom == NULL)
  {
    delete iluacom;
    lua_pushnil();
    return;
  }

  tLuaCOM::lbeans->push(lcom);
}

/*
 *  lc_CLSIDfromProgId
 *  Retorna string contendo CLSID associado a um ProgID
 */

static void lc_CLSIDfromProgId()
{
   char* str = lua_getstring(lua_getparam(1));
   wchar_t* clsid_str;
   wchar_t* progId;
   CLSID clsid;
   HRESULT hr;

   progId = (wchar_t*) malloc( (strlen(str) + 1) * sizeof(wchar_t));
   mbstowcs(progId,str,strlen(str)+1);

   hr = CLSIDFromProgID(progId, &clsid);
   free(progId);
   if (FAILED(hr))
   {
      cout << "Error looking for class " << str << endl;
      lua_pushnil();
      return;
   }
   hr = StringFromCLSID(clsid, &clsid_str);
   if (FAILED(hr))
   {
      cout << "Error looking for prog_id"<< endl;
      lua_pushnil();
      return;
   }

   char* id_str = (char*) malloc( (wcslen(clsid_str) + 1) * sizeof(char));
   wcstombs(id_str,clsid_str,wcslen(clsid_str)+1);
   lua_pushstring(id_str);
   free(id_str);
}

/*
 *  lc_ProgIdfromCLSID
 *  Retorna string contendo o ProgID associado a um CLSID
 */

static void lc_ProgIdfromCLSID()
{
   char* str = lua_getstring(lua_getparam(1));
   wchar_t* clsid_str;
   LPOLESTR progId;
   CLSID clsid;
   HRESULT hr;

   clsid_str = (wchar_t*) malloc( (strlen(str) + 1) * sizeof(wchar_t));
   mbstowcs(clsid_str,str,strlen(str)+1);

   hr = CLSIDFromString(clsid_str, &clsid);
   free(clsid_str);
   if (FAILED(hr))
   {
      cout << "Error looking for class " << str << endl;
      lua_pushnil();
      return;
   }
   hr = ProgIDFromCLSID(clsid, &progId);
   if (FAILED(hr))
   {
      cout << "Error looking for prog_id"<< endl;
      lua_pushnil();
      return;
   }

   char* id_str = (char*) malloc( (wcslen(progId) + 1) * sizeof(char));
   wcstombs(id_str,progId,wcslen(progId)+1);
   lua_pushstring(id_str);
   free(id_str);
}

/*
 *  lc_CreateObject
 *  Retorna um objeto LuaCOM que instancia o objeto
 *  COM identificado pelo ProgID dado
 */

static void lc_CreateObject(void)
{
   char* progId     = NULL;
   wchar_t* olestr  = NULL;
   CLSID clsid;
   HRESULT hr       = S_OK;
   LPUNKNOWN punk   = NULL;
   LPDISPATCH pdisp = NULL;

   if(!LuaAux::getStringParam(1, &progId))
   {
     cout << "lc_CreateObject: wrong parameter" << endl;
     lua_pushnil();
     return;
   }

   olestr = (wchar_t*) malloc( (strlen(progId) + 1) * sizeof(wchar_t));
   assert(olestr);

   mbstowcs(olestr,progId,strlen(progId)+1);
   hr = CLSIDFromProgID(olestr, &clsid);

   free(olestr);

   if (FAILED(hr))
   {
      cout << "ProgID " << progId << " not found" << endl;
      lua_pushnil();
      return;
   }

   hr = CoCreateInstance(clsid,NULL,CLSCTX_ALL, IID_IUnknown,(void**)&punk);
   if (FAILED(hr))
   {
      cout << "Error creating an object of the class " << progId << endl;
      lua_pushnil();
      return;
   }

  tLuaCOM* lcom = tLuaCOM::CreateLuaCOM(punk);

  punk->Release();

  if(lcom == NULL)
  {
    lua_pushnil();
    return;
  }

  tLuaCOM::lbeans->push(lcom);
}


/*
 *  lc_GetObject
 *  Cria um objeto LuaCOM associado a uma instancia
 *  ja' existente do objeto COM identificado pelo 
 *  ProgID dado
 */

static void lc_GetObject(void)
{
   char* progId = lua_getstring(lua_getparam(1));
   wchar_t* olestr;
   CLSID clsid;
   HRESULT hr;
   LPUNKNOWN punk = NULL;
   LPDISPATCH pdisp = NULL;

   olestr = (wchar_t*) malloc( (strlen(progId) + 1) * sizeof(wchar_t));
   mbstowcs(olestr,progId,strlen(progId)+1);
   hr = CLSIDFromProgID(olestr, &clsid);
   free(olestr);
   if (FAILED(hr))
   {
      cout << "Error looking for class " << progId << endl;
      lua_pushnil();
      return;
   }

   hr = GetActiveObject(clsid,NULL,&punk);
   if (FAILED(hr))
   {
      cout << "Error getting an object of the class " << progId << endl;
      lua_pushnil();
      return;
   }

  tLuaCOM* lcom = tLuaCOM::CreateLuaCOM(punk);

  punk->Release();

  if(lcom == NULL)
  {
    lua_pushnil();
    return;
  }

  tLuaCOM::lbeans->push(lcom);
}

/*
 *  lc_addConnection
 *  Associa um connection point de um objeto COM com
 *  outro objeto COM
 */

static void lc_addConnection()
{
  tLuaCOM* server = (tLuaCOM*) 
    tLuaCOM::lbeans->from_lua(lua_getparam(1));
  tLuaCOM* client = (tLuaCOM*) 
    tLuaCOM::lbeans->from_lua(lua_getparam(2));

  if(server == NULL || client == NULL)
  {
    lua_pushnil();
    return;
  }

  bool result = server->addConnection(client);

  if(result == false)
    lua_pushnil();

  // tudo OK
  lua_pushnumber(1);
}

/*
 *  lc_releaseConnection
 *  Desfaz connection point associado a um objeto LuaCOM
 */

static void lc_releaseConnection()
{
  lua_Object table = lua_getparam(1);
  tLuaCOM* lcom = (tLuaCOM*) tLuaCOM::lbeans->from_lua(table);

  lcom->releaseConnection();
}

//
// lc_isMember
//
//  Informa se existe algum metodo ou propriedade com
//  o nome passado como parametro em lua
//

static void lc_isMember(void)
{
  // objeto luacom
  tLuaCOM* lcom = (tLuaCOM*) 
    tLuaCOM::lbeans->from_lua(lua_getparam(1));

  // nome a ser verificado
  char* member_name = lua_getstring(lua_getparam(2));

  if(!lcom || !member_name)
  {
    lua_pushnil();
    return;
  }

  if(lcom->isMember(member_name))
    lua_pushnumber(1);
  else
    lua_pushnil();
}



////////////////////////////////////
//                                //
//  TAG METHODS DO OBJETO LUACOM  //
//                                //
////////////////////////////////////


//
// funcoes auxiliares
//


/*
 * tag method que gerencia garbage collection
 */

static void gc_fb(void)
{
   tLuaCOM* lcom = (tLuaCOM*) lua_getuserdata(lua_getparam(1));

   assert(lcom);
   if(lcom == NULL)
     return;

   delete lcom;
}

/*
 * tag method que gerencia atribuicoes a campos
 * do objeto luacom
 */

static void settable_fb(void)
{
   lua_Object table = lua_getparam(1);
   lua_Object index = lua_getparam(2);
   lua_Object val = lua_getparam(3);

   try
   {
     tLuaCOM* lcom = (tLuaCOM*) tLuaCOM::lbeans->from_lua(table);

     if (lua_isstring(index) && lcom != NULL)
     {
        char* name = lua_getstring(index);
        DISPID dispid;
        HRESULT hr = lcom->get_dispid(name, &dispid);

        if (!FAILED(hr))
        {
           FUNCDESC* pfuncdescr = NULL;

           hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYPUT,&pfuncdescr );

           if(FAILED(hr))
             hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYPUTREF,&pfuncdescr );

           if ( !FAILED(hr) )
           {
             int result = lcom->putprop(pfuncdescr,val);
             lcom->ReleaseFuncDesc(pfuncdescr);

             // se chamada foi bem-sucedida, retorna
             if (result)
               return;
           }
        }
     }
   }
   catch(class tLuaCOMException&)
   {
     // por enquanto, ignora excecoes
   }

   lua_pushobject(table);
   lua_pushobject(index);
   lua_pushobject(val);
   lua_rawsettable();
}



/*
 * Checa se o parametro self foi passado corretamente
 */ 

static bool check_self(lua_Object self)
{
  // garante existencia e garante que o parametro e' uma tabela
  // ou um userdata

  if(self == LUA_NOOBJECT || lua_isnil(self) || 
     !( lua_istable(self) || lua_isuserdata(self) )
    )
  {
    return false;
  }

  return true;
}



/*
 * Closure que gerencia chamadas de metodos
 *
 * In: userdata (tLuaCOM *), userdata (FUNCDESC *),
 *     table | userdata, ...
 */
static void lc_call()
{
  tLuaCOM* lcom       = NULL;
  HRESULT hr          = S_OK;
  DISPID dispid       = -1;
  FUNCDESC *pFuncDesc = NULL;

  // posicao dos parametros
  const int luacom_obj_param  = 1;
  const int dispid_param      = 2;
  int user_param_pos          = -1; // precisa ser inicializado
  
  try
  {
    // obtem objeto luacom

    lcom = (tLuaCOM *) lua_getuserdata(lua_getparam(luacom_obj_param));

    CHECK(lcom, INTERNAL_ERROR);

    // pega o dispid
    dispid = (DISPID) lua_getuserdata(lua_getparam(dispid_param));

    // Testa para o caso de ser metodo
    hr = lcom->get_funcdescr(dispid,INVOKE_FUNC,&pFuncDesc);

    if (SUCCEEDED(hr))  //e' metodo
    {
      // Forca usuario a usar : na chamada a metodo, o que
      // faz com que lua passe o argumento "self". Esse argumento
      // nao e' utilizado na implementacao, mas e' exigido para
      // garantir uniformidade sintatica
      if(!check_self(lua_getparam(dispid_param+1)))
      {
        lua_error("Method call without ':'");
        return;
      }

      lcom->call(pFuncDesc, tLuaObjList(dispid_param+2));

      lcom->ReleaseFuncDesc(pFuncDesc);

      return; // fim do processo
    }    

    //// tenta para o caso de ser propget ou propput parametrizado

    // primeiro, pega a lista de parametros, para poder utilizar
    // o numero de parametros passados como discriminador entre
    // propget e propput

    tLuaObjList &params = tLuaObjList(dispid_param+1);

    // testa o propget

    hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYGET,&pFuncDesc);    

    if(SUCCEEDED(hr) && pFuncDesc->cParams == params.get_num_elements())
    {
      // e' propget

      lcom->call(pFuncDesc, params);

      lcom->ReleaseFuncDesc(pFuncDesc);

      return;
    }

    // tenta o propput (ou propputref) parametrizados

    hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYPUT,&pFuncDesc);

    if(FAILED(hr)) // tenta propputref
      hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYPUTREF,&pFuncDesc);

    if(SUCCEEDED(hr) && pFuncDesc->cParams == params.get_num_elements()) 
    {
      lcom->call(pFuncDesc, params);

      lcom->ReleaseFuncDesc(pFuncDesc);

      return;
    }
  }
  catch(class tLuaCOMException&)
  {
    // ignora excecoes
  }

  return;
}


/*
 * tag method que gerencia leituras de campos
 * do objeto luacom
 *
 * In: table, index
 */

static void index_fb(void)
{
  // variaveis utilizadas
  DISPID dispid;
  HRESULT hr              = S_OK;
  FUNCDESC* pfuncdescr    = NULL;
  tLuaCOM* lcom           = NULL;

  try
  {
    // obtem objeto luacom a partir da tabela
    {
      lua_Object table = lua_getparam(1);
      lcom = (tLuaCOM*) tLuaCOM::lbeans->from_lua(table);

      CHECK(lcom, INTERNAL_ERROR);
    }

    // obtem campo acessado pelo programa
    {
      lua_Object index = lua_getparam(2);

      CHECK(lua_isstring(index), INTERNAL_ERROR);

      // tenta obter dispid correspondente ao nome fornecido
      char *name = lua_getstring(index);
      hr = lcom->get_dispid(name, &dispid);

      // se nome fornecido nao corresponder a nenhum dispid...
      if (FAILED(hr)) 
        return;
    }

    // se for propget sem parametro, executa direto

    hr = lcom->get_funcdescr(dispid,INVOKE_PROPERTYGET,&pfuncdescr);

    if (SUCCEEDED(hr) && pfuncdescr->cParams == 0)  
    {
      //e' propget sem parametros. Executa direto

      lcom->getprop(pfuncdescr);
    }
    else // repassa para lc_call decidir o que deve ser chamado
    {
      lua_pushuserdata((void *) lcom);
      lua_pushuserdata((void *) dispid);
      lua_pushcclosure(lc_call,2);
    }

    lcom->ReleaseFuncDesc(pfuncdescr);
  }
  catch(class tLuaCOMException&)
  {
    // ignora excecoes proprias
  }

  return;
}


/////////////////////////////////////////////
//                                         //
//  TABELA DAS FUNCOES EXPORTADAS PARA LUA //
//                                         //
/////////////////////////////////////////////

static struct luaL_reg functions_tb []= 
{
  {"luacom_CreateObject",lc_CreateObject},
  {"luacom_GetObject",lc_GetObject},
  {"luacom_CLSIDfromProgId",lc_CLSIDfromProgId},
  {"luacom_ImplInterface",lc_ImplInterface},
  {"luacom_ImplInterfaceFromTypelib",lc_ImplInterfaceFromTypelib},
  {"luacom_addConnection", lc_addConnection},
  {"luacom_releaseConnection", lc_releaseConnection},
  {"luacom_ProgIdfromCLSID",lc_ProgIdfromCLSID},
  {"luacom_isMember",lc_isMember},
  {"luacom_Connect",lc_Connect},
  {"luacom_ShowHelp",lc_ShowHelp}
};
  
static struct luaL_reg tagmethods_tb[]= {{"index", index_fb},
                                  {"settable",settable_fb}};
static struct luaL_reg ud_tagmethods_tb[]= {{"gc", gc_fb}};


/////////////////////////////////////
//                                 //
//  FUNCOES EXPORTADAS PARA C/C++  //
//                                 //
/////////////////////////////////////

/*
 *  luacom_idispatch2luacom
 *  Recebe um ponteiro void* que deve ser
 *  um ponteiro para uma implementacao de uma
 *  interface IDispatch. E' criado um objeto
 *  LuaCOM cuja implementacao e' dada por essa
 *  interface. O objeto criado e' colocado na
 *  pilha de Lua
 */

void luacom_IUnknown2LuaCOM(void *punk_arg)
{
  unsigned int ninfo = 0;
  IUnknown *punk = (IUnknown *) punk_arg;

  assert(punk);
  if(punk == NULL)
  {
    lua_pushnil();
    return;
  }

  tLuaCOM* lcom = tLuaCOM::CreateLuaCOM(punk);

  if(lcom == NULL)
  {
    lua_pushnil();
    return;
  }

  tLuaCOM::lbeans->push(lcom);
}

/*
 *  Inicializacao da biblioteca
 */

int luacom_open(void)
{
  //printf("LuaCOM Revision: %s\n\n", g_version);

  if (OleInitialize(NULL) != 0)
    return 0;

  luaL_openlib(functions_tb,sizeof(functions_tb)/sizeof(struct luaL_reg));

  tLuaCOM::lbeans = LuaBeans::createBeans(tLuaCOM::TAG);
  tLuaCOM::lbeans->reg_tagmethods(tagmethods_tb,
                        sizeof(tagmethods_tb)/sizeof(struct luaL_reg));
  tLuaCOM::lbeans->reg_ud_tagmethods(ud_tagmethods_tb,
                          sizeof(ud_tagmethods_tb)/sizeof(struct luaL_reg));

  // inicializa valores booleanos para uso dos metodos
  // de automacao OLE

  lua_pushnumber(LUACOM_TRUE);
  lua_setglobal("LCTRUE");

  lua_pushnumber(LUACOM_FALSE);
  lua_setglobal("LCFALSE");

  return 1;
}

/*
 *  Fechamento da biblioteca
 */

void luacom_close(void)
{
   OleUninitialize();
}







