// tLuaCOMTypeHandler.cpp: implementation of the tLuaCOMTypeHandler class.
//
//////////////////////////////////////////////////////////////////////

// RCS Info
static char *rcsid = "$Id: tLuaCOMTypeHandler.cpp,v 1.11 2001-05-18 08:59:04-03 almendra Exp almendra $";
static char *rcsname = "$Name:  $";


#include <ole2.h>

extern "C"
{
#include <lua.h>
}

#include <iostream.h>
#include <assert.h>
#include <stdio.h>

#include "tLuaCOMTypeHandler.h"
#include "tLuaCOM.h"
#include "tLuaVector.h"
#include "tLuaCOMException.h"

#include "tUtil.h"
#include "LuaAux.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

tLuaCOMTypeHandler::tLuaCOMTypeHandler(ITypeInfo *ptypeinfo)
{
  m_typeinfo = ptypeinfo;
  m_typeinfo->AddRef();
}

tLuaCOMTypeHandler::~tLuaCOMTypeHandler()
{
  m_typeinfo->Release();
  m_typeinfo = NULL;
}

bool tLuaCOMTypeHandler::com2lua(VARIANTARG varg)
{
  VARIANTARG new_varg;
  
  VariantInit(&new_varg);

  // trata SAFE ARRAY separadamente
  if(varg.vt & VT_ARRAY)
  {
    return safearray_com2lua(varg);
  }

  switch (varg.vt)
  {
  case VT_VOID:
  case VT_EMPTY:
    // nao faz nada
    break;

  case VT_NULL:
    // NULL do SQL.
    lua_pushnil();
    break;

  case VT_CY:
  case VT_CY | VT_BYREF:
  case VT_UI1:
  case VT_UI1 | VT_BYREF:
  case VT_UI2:
  case VT_UI2 | VT_BYREF:
  case VT_UI4:
  case VT_UI4 | VT_BYREF:
  case VT_INT:
  case VT_INT | VT_BYREF:
  case VT_UINT:
  case VT_UINT | VT_BYREF:
  case VT_I1:
  case VT_I1 | VT_BYREF:
  case VT_I2:
  case VT_I2 | VT_BYREF:
  case VT_I4:
  case VT_I4 | VT_BYREF:
  case VT_R4:
  case VT_R4 | VT_BYREF:
  case VT_R8:
  case VT_R8 | VT_BYREF:
    {
      new_varg.vt = VT_R8;
      HRESULT hr = VariantChangeType(&new_varg, &varg, 0, VT_R8);
      assert(SUCCEEDED(hr));

      lua_pushnumber(new_varg.dblVal);

      break;
    }

  case VT_DATE:
  case VT_DATE | VT_BYREF:
    {
      new_varg.vt = VT_BSTR;
      HRESULT hr = VariantChangeType(&new_varg, &varg, 0, VT_BSTR);
      assert(SUCCEEDED(hr));

      lua_pushstring((char *) tUtil::bstr2string(new_varg.bstrVal));

      break;
    }


  case VT_ERROR: // assume que omitiu parametro
    lua_pushnil();
    break;

  case VT_BOOL:
    if (varg.boolVal)
       lua_pushnumber(LUACOM_TRUE);
    else
       lua_pushnumber(LUACOM_FALSE);
    break;

  case VT_BOOL | VT_BYREF:
    if (*varg.pboolVal)
       lua_pushnumber(LUACOM_TRUE);
    else
       lua_pushnumber(LUACOM_FALSE);
    break;

  case VT_BSTR:
    {
      const char* str = tUtil::bstr2string(varg.bstrVal);
      lua_pushstring((char *) str);

      break;
    }
  case VT_BSTR | VT_BYREF:
    {
      const char* str = tUtil::bstr2string(*varg.pbstrVal);
      lua_pushstring((char *) str);

      break;
    }

  case VT_DISPATCH:
    {
      unsigned int ninfo = 0;
      IDispatch *pdisp = varg.pdispVal;

      tLuaCOM* lcom = tLuaCOM::CreateLuaCOM(pdisp);

      if(lcom == NULL)
      {
        lua_pushnil();
        break;
      }

      tLuaCOM::lbeans->push(lcom);
    }
    break;

  case VT_UNKNOWN:
  default:
    lua_error("Type not implemented");
  }

  return true;
}

bool tLuaCOMTypeHandler::lua2com(const TYPEDESC & tdesc,
                                 lua_Object luaval,
                                 VARIANTARG * pvarg,
                                 bool variant_initialized)
{
  assert(luaval != LUA_NOOBJECT);
  if(luaval == LUA_NOOBJECT)
    return false;

  VARTYPE vt;

  if(!variant_initialized)
    vt = tdesc.vt;
  else
    vt = pvarg->vt;

   switch (vt)
   {
   case VT_UI1:
   case VT_I2:
   case VT_I4:
   case VT_R4:
   case VT_R8:
   case VT_CY:
   case VT_DATE:
   case VT_INT:
     {
       FromNumberToVariant(pvarg, luaval, tdesc.vt);
       
       break;
     }

   case VT_UI1 | VT_BYREF:
     {
       if(!variant_initialized)
       {
         pvarg->vt = VT_UI1 | VT_BYREF;
         pvarg->pbVal = new unsigned char;
       }

       *(pvarg->pbVal) = (unsigned char)lua_getnumber(luaval);

       break;
     }

   case VT_I2 | VT_BYREF:
     {
       if(!variant_initialized)
       {
         pvarg->vt = VT_I2 | VT_BYREF;
         pvarg->piVal = new short;
       }

       *(pvarg->piVal) = (short)lua_getnumber(luaval);

       break;
     }

   case VT_I4 | VT_BYREF:
     {
       if(!variant_initialized)
       {
        pvarg->vt = VT_I4 | VT_BYREF;
        pvarg->plVal = new long;
       }

       *(pvarg->plVal) = (long)lua_getnumber(luaval);

       break;
     }

   case VT_R4 | VT_BYREF:
     {
       if(!variant_initialized)
       {
        pvarg->vt = VT_R4 | VT_BYREF;
        pvarg->pfltVal = new float;
       }

       *(pvarg->pfltVal) = (float)lua_getnumber(luaval);

       break;
     }

   case VT_R8 | VT_BYREF:
     {
       if(!variant_initialized)
       {
        pvarg->vt = VT_R8 | VT_BYREF;
        pvarg->pdblVal = new double;
       }

       *(pvarg->pdblVal) = (double)lua_getnumber(luaval);

       break;
     }

   case VT_CY | VT_BYREF:
     {
       VARIANTARG varg;

       FromNumberToVariant(&varg, luaval, VT_CY);

       pvarg->vt = VT_CY | VT_BYREF;
       *pvarg->pcyVal = varg.cyVal;

       break;
     }

   case VT_DATE | VT_BYREF:
     {
       VARIANTARG varg;

       FromNumberToVariant(&varg, luaval, VT_DATE);

       pvarg->vt = VT_DATE | VT_BYREF;
       *pvarg->pdate = varg.date;

       break;

     }
   
   case VT_INT | VT_BYREF:
     {
       if(!variant_initialized)
       {
        pvarg->vt = VT_INT | VT_BYREF;
        pvarg->pintVal = new int;
       }

       *pvarg->pintVal = (int) lua_getnumber(luaval);

       break;
     }

   case VT_BOOL:
   case VT_BOOL | VT_BYREF:
     {
       FromBoolToVariantarg(pvarg, luaval, tdesc.vt);

       break;
     }

   case VT_BSTR:
     {
       pvarg->vt = VT_BSTR;
       pvarg->bstrVal = tUtil::string2bstr(lua_getstring(luaval));

       break;
     }

   case VT_BSTR | VT_BYREF:
     {
       if(!variant_initialized)
       {
        pvarg->vt = VT_BSTR | VT_BYREF;
        pvarg->pbstrVal = new BSTR;
       }

       *pvarg->pbstrVal = tUtil::string2bstr(lua_getstring(luaval));

       break;
     }

    case VT_VARIANT:
      {
        if (lua_isnumber(luaval))
        {
           pvarg->vt = VT_R8;
           pvarg->dblVal = lua_getnumber(luaval);
        }
        else if (lua_isstring(luaval))
        {
          pvarg->vt = VT_BSTR;
          pvarg->bstrVal = tUtil::string2bstr(lua_getstring(luaval));
        }
        else if(tLuaCOM::lbeans->from_lua(luaval) != NULL)
        {
          pvarg->vt = VT_DISPATCH;

          FromDispatchToVariantarg(pvarg, luaval, pvarg->vt);
        }
        else
        {
          lua_error("Unable to convert lua type to COM type");
        }

        break;
      }

   case VT_DISPATCH:
     {
       FromDispatchToVariantarg(pvarg, luaval, tdesc.vt);

       break;
     }

   case VT_PTR:
     {
       return from_PTR(tdesc, luaval, pvarg);
       break;
     }

   case VT_USERDEFINED:
     {
       if(is_enum(tdesc))
       {
         TYPEDESC int_type;
         int_type.vt = VT_INT;

         return lua2com(int_type, luaval, pvarg, false);
       }
       else
         lua_error("Could not understand user-defined type");

       break;

     }

   case VT_SAFEARRAY:
     {
       return safearray_lua2com(*tdesc.lptdesc, luaval, pvarg);
       break;
     }

   case VT_VOID:
     // nao faz nada
     break;

   default:
      lua_error("Type not implemented");
   }

   return true;
}

bool tLuaCOMTypeHandler::setRetval(const FUNCDESC * funcdesc,
                                   lua_Object luaval,
                                   VARIANTARG * pvarg)
{
  int num_args = funcdesc->cParams;
  bool found_retval = false;

  while(num_args)
  {
    if(funcdesc->lprgelemdescParam[num_args - 1].idldesc.wIDLFlags & 
        (IDLFLAG_FOUT | IDLFLAG_FRETVAL)
        )
    {
      lua2com(
        funcdesc->lprgelemdescParam[num_args - 1].tdesc,
        luaval,
        pvarg,
        false);
      found_retval = true;
      break;
    }
    num_args--;
  }

  // Se nao encontrou nos parametros de saida, tenta
  // obte-lo do tipo da funcao
  if(!found_retval && 
      funcdesc->elemdescFunc.tdesc.vt != VT_VOID &&
      funcdesc->elemdescFunc.tdesc.vt != VT_HRESULT 
      )
  {
    lua2com(funcdesc->elemdescFunc.tdesc, luaval, pvarg, false);
    found_retval = true;
  }
  else
    return false;

  return true;
}


void tLuaCOMTypeHandler::pushOutValues(tLuaObjList& params,
                                      const FUNCDESC * funcdesc,
                                      const DISPPARAMS& dispparams)
{
  const int num_args = funcdesc->cParams;
  int i = 0;
   
  // Procura valor de retorno dos parametros de saida
  for(i = 0; i < num_args; i++)
  {
    // aliases para simplificar digitacao
    const USHORT& paramflags =
      funcdesc->lprgelemdescParam[i].paramdesc.wParamFlags;

    const VARTYPE& vt = 
      funcdesc->lprgelemdescParam[i].tdesc.vt;

    if( ( paramflags == PARAMFLAG_NONE && (vt == VT_PTR) ) || 
        ( paramflags & PARAMFLAG_FOUT )
      )
    {
      // value
      com2lua(dispparams.rgvarg[num_args - i - 1]);
    }
  }
}


//
// Faz conversao lua2com do tipo VT_PTR
//

bool tLuaCOMTypeHandler::from_PTR(const TYPEDESC & tdesc, lua_Object luaval, VARIANTARG * pvarg)
{
  TYPEDESC pointed_at;

  // continua indirecao
  pointed_at.vt = tdesc.lptdesc->vt;
  pointed_at.lptdesc = tdesc.lptdesc->lptdesc;

  // VT_DISPATCH e VT_UNKNOWN possuem valores que sao
  // naturalmente ponteiro, logo nao precisam de BYREF
  // ( IDispatch * ou IUnknown * )
  if(pointed_at.vt == VT_DISPATCH || pointed_at.vt == VT_UNKNOWN)
  {
    // nao faz nada aqui
  }
  else if(pointed_at.vt == VT_PTR)
  {
    // continua indirecao mais uma vez
    pointed_at.vt = pointed_at.lptdesc->vt;
    pointed_at.lptdesc = pointed_at.lptdesc->lptdesc;

    // so' ha' dupla indirecao para dispatch e unknown
    assert(pointed_at.vt == VT_DISPATCH ||
           pointed_at.vt == VT_UNKNOWN);

    if(!(pointed_at.vt == VT_DISPATCH || 
      pointed_at.vt == VT_UNKNOWN))
      return false;

    // agora precisamos de VT_BYREF, pois estamos tratando de
    // IDispatch ** ou IUnknown **
    pointed_at.vt |= VT_BYREF; 
  }
  else if(pointed_at.vt == VT_USERDEFINED)
  {
    // se for enumeracao, continua
    if(is_enum(pointed_at))
      pointed_at.vt = VT_INT | VT_BYREF;
    else
      lua_error("Not supported userdefined");
  }
  else // outros tipos tem tratamento padrao
  {
    pointed_at.vt |= VT_BYREF; 
  }

  return lua2com(pointed_at, luaval, pvarg, false);
}


//
// Inicializa parametro de saida
//

void tLuaCOMTypeHandler::initOutParameter(VARIANTARG & varg, const TYPEDESC & tdesc)
{
  switch(tdesc.vt)
  {

    //
    // Tipos que podem aparecer num VARIANT (inicializacao direta)
    //

  case VT_I2 | VT_BYREF:   // 2-byte signed int.
    {
      varg.vt = VT_I2 | VT_BYREF;
      varg.piVal = new short(0);

      break;
    }

  case VT_I4 | VT_BYREF:   // 4-byte-signed int.
    {
      varg.vt = VT_I4 | VT_BYREF;
      varg.plVal = new long(0);

      break;
    }

  case VT_R4 | VT_BYREF:		// 4-byte real. 
    {
      varg.vt = VT_R4 | VT_BYREF;
      varg.pfltVal = new float(0);

      break;
    }

  case VT_R8 | VT_BYREF:		// 8-byte real.
    {
      varg.vt = VT_R8 | VT_BYREF;
      varg.pdblVal = new double(0);

      break;
    }

  case VT_CY | VT_BYREF:		// Currency.
    {
      varg.vt = VT_CY | VT_BYREF;
      varg.pcyVal = new CY;

      break;
    }

  case VT_DATE | VT_BYREF:		// Date.
    {
      varg.vt = VT_DATE | VT_BYREF;
      varg.pdate = new DATE;

      break;
    }

  case VT_BSTR | VT_BYREF:		// Automation string.
    {
      varg.vt = VT_BSTR | VT_BYREF;
      varg.pbstrVal = new BSTR;

      *varg.pbstrVal = SysAllocString(NULL);

      break;
    }

  case VT_ERROR | VT_BYREF:		// Scodes.
    {
      varg.vt = VT_ERROR | VT_BYREF;
      varg.pscode = new SCODE(DISP_E_PARAMNOTFOUND);

      break;
    }

  case VT_BOOL | VT_BYREF:		// Boolean; True=-1, False=0.
    {
      varg.vt = VT_BOOL | VT_BYREF;
      varg.pboolVal = new VARIANT_BOOL(FALSE);

      break;
    }

  case VT_VARIANT | VT_BYREF:		// VARIANT FAR*.
    {
      varg.vt = VT_VARIANT | VT_BYREF;
      varg.pvarVal = new VARIANT;

      VariantInit(varg.pvarVal);

      break;
    }

  case VT_UI1 | VT_BYREF:		// Unsigned char.
    {
      varg.vt = VT_UI1 | VT_BYREF;
      varg.pbVal = new unsigned char;

      break;
    }

  case VT_PTR:				// Pointer type.
    {
      TYPEDESC pointed_at;

      // continua indirecao
      pointed_at.vt = tdesc.lptdesc->vt;
      pointed_at.lptdesc = tdesc.lptdesc->lptdesc;

      // VT_DISPATCH e VT_UNKNOWN possuem valores que sao
      // naturalmente ponteiro, logo nao precisam de BYREF
      // ( IDispatch * ou IUnknown * )
      if(pointed_at.vt == VT_DISPATCH || pointed_at.vt == VT_UNKNOWN)
      {
        // nao faz nada aqui
      }
      else if(pointed_at.vt == VT_PTR)
      {
        // continua indirecao mais uma vez
        pointed_at.vt = pointed_at.lptdesc->vt;
        pointed_at.lptdesc = pointed_at.lptdesc->lptdesc;

        // so' ha' dupla indirecao para dispatch e unknown
        assert(pointed_at.vt == VT_DISPATCH ||
               pointed_at.vt == VT_UNKNOWN);

        if(!(pointed_at.vt == VT_DISPATCH || 
          pointed_at.vt == VT_UNKNOWN))
          return;

        // agora precisamos de VT_BYREF, pois estamos tratando de
        // IDispatch ** ou IUnknown **
        pointed_at.vt |= VT_BYREF; 
      }
      else if(pointed_at.vt == VT_USERDEFINED)
      {
        if(is_enum(pointed_at))
          pointed_at.vt = VT_INT | VT_BYREF;
        else
          return;
      }
      else // outros tipos tem tratamento padrao
      {
        pointed_at.vt |= VT_BYREF; 
      }

      initOutParameter(varg, pointed_at);

      break;
    }

    // Daqui para baixo so' e' acessado em chamadas recursivas
    // (via VT_PTR)

    case VT_DISPATCH | VT_BYREF:
    {
      varg.vt = VT_DISPATCH | VT_BYREF;
      varg.ppdispVal = new IDispatch*;
      *varg.ppdispVal = NULL;

      break;
    }

    case VT_UNKNOWN | VT_BYREF:		// IUnknown FAR*.
    {
      varg.vt = VT_UNKNOWN | VT_BYREF;
      varg.ppunkVal = new IUnknown*;
      *varg.ppunkVal = NULL;

      break;
    }

    // tipo utilizado nas enumeracoes
    case VT_INT | VT_BYREF:
      {
        varg.vt = VT_INT | VT_BYREF;
        varg.pintVal = new int;
        *varg.pintVal = 0;
        break;
      }

  default:
    {
      // Aqui entram todos os tipos que nao sao passados por
      // referencia e que nao precisam de inicializacao
      break;
    }
  }

}

void tLuaCOMTypeHandler::releaseVariants(DISPPARAMS *pDispParams)
{
  unsigned int i = 0;
  VARIANTARG* &vargs = pDispParams->rgvarg;

  if (vargs != NULL)
  {
    for (i = 0; i < pDispParams->cArgs; i ++)
    {
      if(vargs[i].vt & VT_BYREF)
      {
        switch(vargs[i].vt & (~VT_BYREF))
        {
        case VT_UI1:
          delete vargs[i].pbVal;
          break;

        case VT_I2:
          delete vargs[i].piVal;
          break;

        case VT_I4:
          delete vargs[i].plVal;
          break;

        case VT_R4:
          delete vargs[i].pfltVal;
          break;

        case VT_R8:
          delete vargs[i].pdblVal;
          break;

        case VT_BSTR:
          SysFreeString(*vargs[i].pbstrVal);
          delete vargs[i].pbstrVal;
          break;

        case VT_DISPATCH:
          delete vargs[i].ppdispVal;
          break;

        case VT_UNKNOWN:
          delete vargs[i].ppunkVal;
          break;

        case VT_DATE:
          delete vargs[i].pdate;
          break;

        case VT_CY:
          delete vargs[i].pcyVal;
          break;

        default: // nao faz nada
          break;
        }
      }
      VariantClear(&vargs[i]);
    }

    delete [] vargs;

    vargs = NULL;
    pDispParams->cArgs = 0;
  }

}

HRESULT tLuaCOMTypeHandler::FromNumberToVariant(VARIANTARG *pvarg, lua_Object luaval, VARTYPE vt)
{
  HRESULT hr = S_OK;
       
  pvarg->dblVal = lua_getnumber(luaval);
  pvarg->vt = VT_R8;

  hr = VariantChangeType(pvarg, pvarg, 0, vt);
  assert(SUCCEEDED(hr));

  return hr;
}

void tLuaCOMTypeHandler::FromBoolToVariantarg(VARIANTARG *pvarg, lua_Object luaval, VARTYPE vt)
{
  assert(vt == VT_BOOL || vt == (VT_BOOL | VT_BYREF));

  if(!(vt == VT_BOOL || vt == (VT_BOOL | VT_BYREF)))
    return;

  VARIANT_BOOL boolVal;

  pvarg->vt = vt;

  if (lua_getnumber(luaval) == LUACOM_FALSE)
    boolVal = VARIANT_FALSE;
  else 
    boolVal = VARIANT_TRUE;

  if(vt & VT_BYREF)
  {
    pvarg->pboolVal = new VARIANT_BOOL;
    *pvarg->pboolVal = boolVal;
  }
  else
  {
    pvarg->boolVal = boolVal;
  }
}

void tLuaCOMTypeHandler::FromDispatchToVariantarg(VARIANTARG *pvarg,
                                                  lua_Object luaval,
                                                  VARTYPE vt)
{
  assert(vt == VT_DISPATCH || vt == (VT_DISPATCH | VT_BYREF));

  if(!(vt == VT_DISPATCH || vt == (VT_DISPATCH | VT_BYREF)))
    return;

  tLuaCOM *lcom = (tLuaCOM *) tLuaCOM::lbeans->from_lua(luaval);

  if(!lcom)
  {
    lua_error("Not an Interface");
  }

  IDispatch *pdisp = lcom->GetIDispatch();
  
  pdisp->AddRef();

  pvarg->vt = vt;

  if(vt & VT_BYREF)
  {
    pvarg->ppdispVal = new IDispatch*;
    *pvarg->ppdispVal = pdisp;
  }
  else
  {
    pvarg->pdispVal = pdisp;
  }
}

//
// Verifica se tipo definido pelo usuario e' um enum
//

bool tLuaCOMTypeHandler::is_enum(const TYPEDESC& typedesc)
{
  HRESULT hr;
  ITypeInfo *userdef = NULL;
  TYPEATTR *typeattr = NULL;
  bool retval = false;

  hr = m_typeinfo->GetRefTypeInfo(typedesc.hreftype, &userdef);

  if(FAILED(hr))
    lua_error("Could not understand user-defined type");

  assert(userdef);

  hr = userdef->GetTypeAttr(&typeattr);

  if(FAILED(hr))
  {
    userdef->Release();
    lua_error("Could not understand user-defined type");
  }
  
  assert(typeattr);

  if(typeattr->typekind == TKIND_ENUM)
    retval = true;
  else
    retval = false;

  userdef->ReleaseTypeAttr(typeattr);
  userdef->Release();

  return retval;
}


//
// Preenche estrutura DISPPARAMS, inicializando parametros
//

void tLuaCOMTypeHandler::fillDispParams(DISPPARAMS& rDispParams,
                                        FUNCDESC * pFuncDesc,
                                        tLuaObjList& params)
{
  static DISPID dispidNamed = DISPID_PROPERTYPUT;

  rDispParams.cArgs = 0;
  rDispParams.cNamedArgs = 0;
  rDispParams.rgvarg = NULL;
  rDispParams.rgdispidNamedArgs = NULL;


  // se funcao nao recebe parametros...
  if (pFuncDesc->cParams == 0)
  {
    return;
  }  
  else // chamada normal
  {
    unsigned short i            = 0;
    bool not_supported_found  = false;
    lua_Object val            = LUA_NOOBJECT;
    unsigned short max_params      = pFuncDesc->cParams;

    // referencias para simplificar nomes
    unsigned int& r_cArgs   = rDispParams.cArgs; 
    VARIANTARG* &r_rgvarg   = rDispParams.rgvarg;



    // caso particular do propertyput
    if(pFuncDesc->invkind == DISPATCH_PROPERTYPUT ||
       pFuncDesc->invkind == DISPATCH_PROPERTYPUTREF)
    {
      rDispParams.cNamedArgs = 1;
      rDispParams.rgdispidNamedArgs = &dispidNamed;
    }

    r_cArgs = 0; // comeca vazio


    // cria array com tamanho suficiente para numero maximo
    // de parametros

    r_rgvarg = new VARIANTARG[max_params]; 
 

    // itera no array lprgelemdescParam procurando pegar
    // os parametros da tabela lua

    for (i = 0; i < max_params && not_supported_found == false; i++)
    {
      VariantInit(&r_rgvarg[r_cArgs]);
      val = params.getparam(r_cArgs);

      switch(pFuncDesc->lprgelemdescParam[i].paramdesc.wParamFlags)
      {
        // casos em que e' esperado um parametro de entrada
      case PARAMFLAG_NONE:
        {
          if(val != LUA_NOOBJECT && !lua_isnil(val)) 
          {
            lua2com(
              pFuncDesc->lprgelemdescParam[i].tdesc,
              val,
              &r_rgvarg[r_cArgs],
              false
              );
          }
          else
          {
            initOutParameter(
              r_rgvarg[r_cArgs],
              pFuncDesc->lprgelemdescParam[i].tdesc);
          }

          r_cArgs++;

          break;
        }

      case PARAMFLAG_FIN:
      case PARAMFLAG_FIN | PARAMFLAG_FOUT:

      case PARAMFLAG_FIN | PARAMFLAG_FOPT:
      case PARAMFLAG_FIN | PARAMFLAG_FOUT | PARAMFLAG_FOPT:
        {
          // se usuario omitiu parametro, deixamos que o objeto decida se
          // esta' errado ou nao

          if(val != LUA_NOOBJECT && !lua_isnil(val)) 
          {
            lua2com(
              pFuncDesc->lprgelemdescParam[i].tdesc,
              val,
              &r_rgvarg[r_cArgs],
              false
              );
          }
          else
          {
            r_rgvarg[r_cArgs].vt = VT_ERROR;
            r_rgvarg[r_cArgs].scode = DISP_E_PARAMNOTFOUND;
          }

          r_cArgs++;

          break;
        }

        // inicializa parametro de saida
      case PARAMFLAG_FOUT:
      case PARAMFLAG_FOUT | PARAMFLAG_FOPT:
        {
          initOutParameter(
            r_rgvarg[r_cArgs],
            pFuncDesc->lprgelemdescParam[i].tdesc);

          r_cArgs++;

          break;
        }
      
        // outros casos nao sao suportados
      default:
        not_supported_found = true;
        break;
      }
    }

    // inverte ordem dos parametros
    if(r_cArgs > 0)
    {
      VARIANTARG temp;

      for(i = 0; i < r_cArgs/2; i++)
      {
        temp = r_rgvarg[i];
        r_rgvarg[i] = r_rgvarg[r_cArgs - i - 1]; 
        r_rgvarg[r_cArgs - i - 1] = temp;
      }
    }

    return;
  }
}


void tLuaCOMTypeHandler::pushLuaArgs(const DISPPARAMS* pDispParams,
                                     const ELEMDESC* pElemDesc)
{
  unsigned int arg = 0;

  for(arg = 0; arg < pDispParams->cArgs; arg++)
  {
    const USHORT& wParamFlags = pElemDesc[arg].paramdesc.wParamFlags;
    const TYPEDESC& tdesc = pElemDesc[arg].tdesc;

    if(wParamFlags & PARAMFLAG_FIN || (wParamFlags == PARAMFLAG_NONE))
    {
      com2lua(pDispParams->rgvarg[pDispParams->cArgs - arg - 1]);
    }
    else if(wParamFlags & (~PARAMFLAG_FRETVAL))
    {
      lua_pushnil();
    }
  }
}

void tLuaCOMTypeHandler::setOutValues(FUNCDESC * pFuncDesc,
                                      DISPPARAMS * pDispParams
                                      )
{
  const int num_args = pFuncDesc->cParams;
  int out_value_pos = 0;
  int i = 0;

  // luacom exige que o usuario retorne algo caso
  // a funcao exija valor de retorno

  if(pFuncDesc->elemdescFunc.tdesc.vt != VT_VOID)
    out_value_pos = 2; // pula valor de retorno
  else
    out_value_pos = 1;
   
  // Procura valor de retorno dos parametros de saida
  for(i = 0; i < num_args; i++)
  {
    // aliases para simplificar digitacao

    const TYPEDESC& tdesc = 
      pFuncDesc->lprgelemdescParam[i].tdesc;

    const USHORT& paramflags =
      pFuncDesc->lprgelemdescParam[i].paramdesc.wParamFlags;


    if((paramflags == PARAMFLAG_NONE 
        && (tdesc.vt == VT_PTR)
        && (tdesc.lptdesc->vt != VT_DISPATCH)
        && (tdesc.lptdesc->vt != VT_UNKNOWN))
      || ( paramflags & PARAMFLAG_FOUT )
      )
    {
      lua_Object outval = lua_getresult(out_value_pos);

      if(outval != LUA_NOOBJECT)
      {
        lua2com(
          tdesc,
          outval,
          &pDispParams->rgvarg[num_args - i - 1],
          true
          );

        out_value_pos++;
      }
      else // acabou lista de valores de retorno
        break;
    }
  }

}

//
// Conversao de Safe Arrays para tabelas lua e vice versa
//

//  funcoes auxiliares

// funcoes auxiliares de safearray_lua2com
namespace safearray_aux
{
  long * dimensionsFromBounds(SAFEARRAYBOUND* bounds, long num_bounds);
  void put_in_array(
    SAFEARRAY* safearray,
    VARIANT var_value,
    long* indices,
    VARTYPE vt);

  void inc_indices(long *indices, SAFEARRAYBOUND *bounds, unsigned long dimensions);
  SAFEARRAYBOUND* getRightOrderedBounds(
    SAFEARRAYBOUND *bounds, 
    unsigned long num_dimensions);
};


SAFEARRAYBOUND* safearray_aux::getRightOrderedBounds(
    SAFEARRAYBOUND *bounds, 
    unsigned long num_dimensions)
{
  SAFEARRAYBOUND* new_bounds = new SAFEARRAYBOUND[num_dimensions];

  unsigned long i = 0;

  for(i = 0; i < num_dimensions; i++)
    new_bounds[i] = bounds[num_dimensions - i - 1];

  return new_bounds;
}


void safearray_aux::put_in_array(SAFEARRAY* safearray,
                         VARIANT var_value,
                         long* indices,
                         VARTYPE safearray_type
                         )
{
  HRESULT hr = S_OK;

  if(safearray_type == VT_VARIANT)
  {
    hr = SafeArrayPutElement(safearray, indices, &var_value);
  }
  else
  {
    switch(var_value.vt)
    {
     case VT_UI1:
     case VT_I2:
     case VT_I4:
     case VT_R4:
     case VT_R8:
     case VT_CY:
     case VT_DATE:
     case VT_INT:
     case VT_BOOL:
      hr = SafeArrayPutElement(safearray, indices, &var_value.dblVal);
      break;

    case VT_BSTR:
    case VT_DISPATCH:
      hr = SafeArrayPutElement(safearray, indices, var_value.bstrVal);
      break;

    default:
      LUACOM_EXCEPTION(INTERNAL_ERROR);
      break;
    }
  }

  if(FAILED(hr))
    LUACOM_EXCEPTION(INTERNAL_ERROR);
}

lua_Object tLuaCOMTypeHandler::get_from_array(SAFEARRAY* safearray,
                                         long *indices,
                                         const VARTYPE& vt
                                         )
{
  VARIANTARG varg;
  void *pv = NULL;

 
  HRESULT hr = S_OK;

  if(vt == VT_VARIANT)
  {
    pv = &varg;
  }
  else
  {
    VariantInit(&varg);
    varg.vt = vt;

    // e' uma union, tanto faz de quem pego o ponteiro
    pv = (void *) &varg.dblVal; 
  }

  hr = SafeArrayGetElement(safearray, indices, pv);

  if(FAILED(hr))
    LUACOM_EXCEPTION(INTERNAL_ERROR);

  bool ok = com2lua(varg);

  VariantClear(&varg);

  if(!ok)
    return LUA_NOOBJECT;

  return lua_pop();
}



void safearray_aux::inc_indices(long *indices, 
                        SAFEARRAYBOUND *bounds,
                        unsigned long dimensions
                        )
{
  unsigned long j = 0;

  indices[0]++;
  j = 0;

  while(
    (indices[j] >= (long) bounds[j].cElements) &&
    (j < (dimensions - 1))
    )
  {
    indices[j] = 0;
    indices[j+1]++;

    j++;
  }
}


//
// Cuida da conversao de tabelas para safe arrays
//

bool tLuaCOMTypeHandler::safearray_lua2com(const TYPEDESC & tdesc,
                                           lua_Object luaval,
                                           VARIANTARG * pvarg)
{
  using namespace safearray_aux;

  CHECK(pvarg, PARAMETER_OUT_OF_RANGE);
  
  // verifica se e' possivel a conversao para arrays
  if(   luaval == LUA_NOOBJECT
     || !lua_istable(luaval)
     )
  {
    return false;
  }


  // cria LuaVector baseado na tabela passada
  tLuaVector luavector;

  if(luavector.InitVectorFromTable(luaval) != tLuaVector::OK)
    return false;

  // Cria variaveis
  unsigned long i = 0;
  const unsigned long dimensions = luavector.get_Dimensions();
  SAFEARRAYBOUND *bounds = new SAFEARRAYBOUND[dimensions];
  SAFEARRAY *safearray = NULL;
  VARIANTARG var_value;

  VariantInit(&var_value);


  // inicializa dimensoes
  for(i = 0; i < dimensions; i++)
  {
    bounds[i].lLbound = 0;
    bounds[i].cElements = luavector.get_Nth_Dimension(dimensions - i);
  }


  // cria array
  safearray = SafeArrayCreate(tdesc.vt, dimensions, bounds);
  
  long *indices = NULL;
  
  try
  {
    CHECK(safearray, INTERNAL_ERROR);

    // Inicializa indices
    indices = new long[dimensions];

    for(i = 0; i < dimensions; i++)
      indices[i] = 0;

    // copia elementos um por um
    while(indices[dimensions - 1] < (long) bounds[dimensions - 1].cElements)
    {
      // obtem valor
      luaval = luavector.getindex(indices, dimensions);

      //converte
      lua2com(tdesc, luaval, &var_value, false);

      // coloca no array
      put_in_array(safearray, var_value, indices, tdesc.vt);

      // incrementa indices
      inc_indices(indices, bounds, dimensions);
    }
  }
  catch(class tLuaVectorException&)
  {
    delete bounds;
    delete indices;

    throw;
  }
  catch(class tLuaCOMException&)
  {
    delete bounds;
    delete indices;

    throw;
  }


  // preenche variantarg
  pvarg->vt = tdesc.vt | VT_ARRAY;
  pvarg->parray = safearray;


  // libera memoria
  delete bounds;
  delete indices;

  return true;
}


long * safearray_aux::dimensionsFromBounds(SAFEARRAYBOUND* bounds,
                                           long num_bounds
                                           )
{
  int i = 0;
  long *dimensions = new long[num_bounds];

  for(i = 0; i < num_bounds; i++)
  {
    dimensions[i] =
      bounds[num_bounds - i - 1].lLbound + bounds[num_bounds - i - 1].cElements; 
  }

  return dimensions;
}



bool tLuaCOMTypeHandler::safearray_com2lua(VARIANTARG & varg)
{
  using namespace safearray_aux;

  CHECK(varg.vt & VT_ARRAY, PARAMETER_OUT_OF_RANGE);

  SAFEARRAYBOUND* bounds = NULL;
  
  try
  {
    SAFEARRAY* safearray = varg.parray;

    // pega dimensoes
    const int num_dimensions = SafeArrayGetDim(safearray);

    bounds = getRightOrderedBounds
      (
      safearray->rgsabound,
      num_dimensions
      );
  
    
      // cria objeto LuaVector
    tLuaVector luavector;

    {
      long *dimensions = dimensionsFromBounds(bounds, num_dimensions);

      try
      {
        if(luavector.InitVectorFromDimensions(dimensions, num_dimensions)
          != tLuaVector::OK)
        {
          delete bounds;
          delete dimensions;

          return false;
        }
      }
      catch(class tLuaVectorException&)
      {
        delete dimensions;
        throw;
      }

      delete dimensions;
    }

    // Inicializa indices
    long *indices = new long[num_dimensions];

    int i = 0;
    for(i = 0; i < num_dimensions; i++)
      indices[i] = 0;

    // extrai tipo de dado do array
    VARTYPE vt = varg.vt & ~VT_ARRAY;

    // guarda objeto lua
    lua_Object luaval = LUA_NOOBJECT;


    // copia elementos um por um
    while(indices[num_dimensions-1] < (long) bounds[num_dimensions-1].cElements)
    {
      // pega do array
      luaval = get_from_array(safearray, indices, vt);

      // seta no luavector
      luavector.setindex(luaval, indices, num_dimensions);

      // incrementa indices
      inc_indices(indices, bounds, num_dimensions);
    }

    // gera tabela a partir do luavector
    lua_Object table = luavector.CreateTable();

    if(table == LUA_NOOBJECT)
      return false;

    lua_pushobject(table);
  }
  catch(class tLuaCOMException&)
  {
    delete bounds;
    return false;
  }
  catch(class tLuaVectorException&)
  {
    delete bounds;
    return false;
  }

  delete bounds;

  return true;
}

