// tLuaVector.cpp: implementation of the tLuaVector class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <assert.h>

#include "tLuaVector.h"
#include "tLuaCOM.h"

char *tLuaVectorException::messages[] =  
     {
       "Internal Error",
       "Index out of bounds",
       "Parameter(s) out of bounds",
       "Precondition Failed"
     };

#define THROW_EXCEPTION(x) throw tLuaVectorException(tLuaVectorException::##x, \
  __FILE__, __LINE__)


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

tLuaVector::tLuaVector()
{
  initialized = false;
}

tLuaVector::~tLuaVector()
{
  // destroi escalares ou luavectors
  freeData();
}


unsigned long tLuaVector::get_Dimensions()
{
  if(length == 0 || elem_type == UNKNOWN)
    throw "get_Dimensions: internal error";


  if(elem_type == SCALAR)
    return 1;
  else
    return 1 + vectors[0]->get_Dimensions();
}

unsigned long tLuaVector::get_Nth_Dimension(unsigned long n)
{
  // precond
  if(length == 0)
    throw "get_Nth_Dimension: internal error";
  if(n == 0)
    throw "get_Nth_Dimension: n should be greater than zero";

  
  if(n == 1)
    return length;
  else if(elem_type == VECTOR)
  {
    return vectors[0]->get_Nth_Dimension(n - 1);
  }
  else
    throw "get_Nth_Dimension: tLuaVector doens't have so many dimensions";

  return 0;
}

lua_Object tLuaVector::getindex(long * indices, unsigned long size)
{
  if(indices == NULL || size == 0)
    throw "tLuaVector::index: assertion failed";

  const long index = indices[size - 1];

  if(index >= (long)length)
    THROW_EXCEPTION(INDEX_OUT_OF_BOUNDS);

  if(size > 1)
  {
    if(elem_type != VECTOR)
      THROW_EXCEPTION(INDEX_OUT_OF_BOUNDS);

    return vectors[indices[size - 1]]->getindex(indices, size - 1);
  }
  else
  {
    return luavals[indices[0]];
  }
}

unsigned long tLuaVector::getLength()
{
  return length;
}

tLuaVector::Type tLuaVector::getType()
{
  return array_type;
}

tLuaVector::Type tLuaVector::lua2Type(lua_Object luaval)
{
  if(lua_isnumber(luaval))
    return NUMBER;
  else if(lua_isstring(luaval))
    return STRING;
  else if(lua_isuserdata(luaval))
    return USERDATA;
  else
    return OTHER;
}

void tLuaVector::freeData()
{
  unsigned long i;

  if(vectors != NULL)
  {
    for(i = 0; i < length; i++)
      delete vectors[i];

    delete vectors;
  }
  else if(luavals != NULL)
  {
    delete luavals;
  }

  initialized = false;

  return;
}

//
// Cria uma tabela Lua a partir da matriz contida no luavector
//
lua_Object tLuaVector::CreateTable(void)
{
  lua_Object table = lua_createtable();

  if(array_type == NOTYPE)
    return LUA_NOOBJECT;

  switch(elem_type)
  {
  case VECTOR:
    {
      unsigned long index = 0;

      for(index = 0; index < length; index++)
      {
        lua_Object child_table = vectors[index]->CreateTable();

        lua_pushobject(table);
        lua_pushnumber(index + 1);

        if(child_table == LUA_NOOBJECT)
          return LUA_NOOBJECT;

        lua_pushobject(child_table);

        lua_settable();
      }
    }

    break;

  case SCALAR:
    {
      unsigned long int index = 0;

      for(index = 0; index < length; index++)
      {
        lua_pushobject(table);
        lua_pushnumber(index + 1);

        if(luavals[index] == LUA_NOOBJECT)
          return LUA_NOOBJECT;

        lua_pushobject(luavals[index]);

        lua_settable();
      }
    }

    break;

  default:
    return LUA_NOOBJECT;
  }

  return table;
}

void tLuaVector::setindex(lua_Object luaval, long * indices, unsigned long size)
{
  if(indices == NULL || size == 0)
    THROW_EXCEPTION(PARAMETER_OUT_OF_BOUNDS);

  const long index = indices[size - 1];

  if(index >= (long)length)
    THROW_EXCEPTION(INDEX_OUT_OF_BOUNDS);

  if(size > 1)
  {
    if(elem_type != VECTOR)
      THROW_EXCEPTION(INDEX_OUT_OF_BOUNDS);

    vectors[indices[size - 1]]->setindex(luaval, indices, size - 1);

    if(array_type == NOTYPE)
      array_type = vectors[indices[size - 1]]->getType();
    else if(vectors[indices[size - 1]]->getType() != array_type)
      array_type = MANY;
  }
  else
  {
    luavals[indices[0]] = luaval;

    if(array_type == NOTYPE)
      array_type = lua2Type(luaval);
    else if(lua2Type(luaval) != array_type)
      array_type = MANY;
  }
}

tLuaVector::tResults tLuaVector::InitVectorFromDimensions(long * dimensions,
                                                          unsigned long num_dimensions
                                                          )
{
  if(initialized == true)
    return ALREADY_INITIALIZED;

  length = 0;
  vectors = NULL;
  luavals = NULL;
  array_type = NOTYPE;
  elem_type = UNKNOWN;
  max_length = 0;

  if(dimensions == NULL || num_dimensions == 0)
    THROW_EXCEPTION(PARAMETER_OUT_OF_BOUNDS);

  if(num_dimensions == 1)
  {
    elem_type = SCALAR;
    length = max_length = dimensions[0];

    luavals = new lua_Object[length];

    {
      unsigned long i = 0;

      for(i = 0; i < length; i++)
        luavals[i] = LUA_NOOBJECT;
    }

    // cria array de luavals e inicializa resto

    initialized = true;
  }
  else
  {
    elem_type = VECTOR;
    length = max_length = dimensions[0];

    vectors = new tLuaVector*[length];

    // inicializa vetores recursivamente

    {
      unsigned long i = 0;

      for(i = 0; i < length; i++)
      {
        vectors[i] = new tLuaVector;
        tResults result = vectors[i]->InitVectorFromDimensions(
          &dimensions[1],
          num_dimensions - 1);

        if(result != OK)
        {
          initialized = false;
          return result;
        }
      }
    }

    initialized = true;
  }

  return OK;
}

tLuaVector::tResults tLuaVector::InitVectorFromTable(lua_Object table)
{
  lua_Object luaval = LUA_NOOBJECT;

  if(initialized)
    return ALREADY_INITIALIZED;

  length = 0;
  vectors = NULL;
  luavals = NULL;
  array_type = NOTYPE;
  elem_type = UNKNOWN;
  max_length = 0;

  // so' podem ser criados vetores a partir
  // de tabelas e de userdata

  if(table == LUA_NOOBJECT ||
      (!lua_istable(table) && !lua_isuserdata(table))
     )
  {
    return NOT_TABLE;
  }


  try
  {
    // Itera em tabela ou userdata, criando LuaVectors recursivamente
    while(1)
    {
      if((length > 0  && elem_type == UNKNOWN) ||
         (length == 0 && elem_type != UNKNOWN))
         THROW_EXCEPTION(INTERNAL_ERROR);

      lua_pushobject(table);

      lua_pushnumber(length + 1);

      luaval = lua_gettable();

      if(luaval == LUA_NOOBJECT || lua_isnil(luaval))
      {
        break;
      }
      
      // se nao for tabela, termina processo de calculo de dimensoes
      // e guarda elemento
      // se for, continua, chamando recursivamente

      if(lua_istable(luaval) && 
         tLuaCOM::lbeans->from_lua(luaval) == NULL)
      {
        // garante que tabela e' uma matriz
        if(elem_type == SCALAR)
          THROW_EXCEPTION(INTERNAL_ERROR);
        else if(elem_type == UNKNOWN)
        {
          elem_type = VECTOR;

          max_length = 10;

          vectors = new tLuaVector*[max_length];
        }

        // realoca array
        if(length == max_length)
        {
          tLuaVector **new_vectors = NULL;

          max_length *= 2;

          new_vectors = new tLuaVector*[max_length];

          unsigned long i = 0;

          for(i = 0; i < length; i++)
            new_vectors[i] = vectors[i];

          delete vectors;
          vectors = new_vectors;
        }

        // obtem novo vetor
        vectors[length] = new tLuaVector;

        tResults result = vectors[length]->InitVectorFromTable(luaval);

        if(result != OK)
        {
          initialized = false;
          return result;
        }

        length++;

        // caso nova tabela tenha dimensao diferente, aborta processo
        // (nao e' uma matriz)
        if(length == 1)
        {
          array_type = vectors[length - 1]->getType();
        }
        else if(
          vectors[length - 1]->getLength() != 
          vectors[length - 2]->getLength())
        {
          THROW_EXCEPTION(INTERNAL_ERROR);
        }
        else if(
          vectors[length - 1]->getType() != 
          vectors[length - 2]->getType())
        {
          array_type = MANY;
        }
      }
      else
      {
        if(elem_type == VECTOR)
          THROW_EXCEPTION(INTERNAL_ERROR);
        else if(elem_type == UNKNOWN)
        {
          elem_type = SCALAR;

          max_length = 10;

          luavals = new lua_Object[max_length];
        }

        // realoca luavals
        if(length == max_length)
        {
          lua_Object *new_luavals = NULL;

          max_length *= 2;

          new_luavals = new lua_Object[max_length];

          unsigned long i = 0;

          for(i = 0; i < length; i++)
            new_luavals[i] = luavals[i];

          delete luavals;
          luavals = new_luavals ;
        }

      
        // guarda luaval
        luavals[length] = luaval;
        length++;

        // verifica se e' do mesmo tipo que o anterior
        if(length > 1 && lua2Type(luaval) != array_type)
        {
          array_type = MANY;
        }
        else if(length == 1)
        {
          array_type = lua2Type(luaval);
        }
      }
    }
  }
  catch(class tLuaVectorException)
  {
    freeData();
    initialized = false;
    throw;
  }

  initialized = true;
    
  return OK;
}
